package com.yd.api.practitioner.service.impl;

import com.yd.api.practitioner.service.ScheduleTrackService;
import com.yd.api.practitioner.vo.organization.OrganizationQueryResponseVO;
import com.yd.api.result.CommonResult;
import com.yd.api.result.JsonResult;
import com.yd.dal.entity.customer.AclPractitionerPotentialAssignedTrack;
import com.yd.dal.entity.marketing.MktLeadsAssignedTrack;
import com.yd.dal.entity.meta.MdDropOptions;
import com.yd.dal.entity.practitioner.ScheduleTrack;
import com.yd.dal.mapper.marketing.ScheduleTrackMapper;
import com.yd.dal.service.customer.AclPractitionerPotentialAssignedTrackDALService;
import com.yd.dal.service.customer.AclPractitionerSubordinateSystemDALService;
import com.yd.dal.service.marketing.MktLeadsAssignedTrackDALService;
import com.yd.dal.service.meta.MdDropOptionsDALService;
import com.yd.util.CommonUtil;
import com.yd.util.config.ZHBErrorConfig;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

@Service
public class ScheduleTrackServiceImpl implements ScheduleTrackService {
    @Autowired
    private ScheduleTrackMapper scheduleTrackMapper;
    @Autowired
    private AclPractitionerPotentialAssignedTrackDALService aclPractitionerPotentialAssignedTrackDALService;
    @Autowired
    private MdDropOptionsDALService mdDropOptionsDALService;
    @Autowired
    private MktLeadsAssignedTrackDALService mktLeadsAssignedTrackDALService;

    @Override
    public JsonResult insert(ScheduleTrack schedule) {
        JsonResult result = new JsonResult();
        result.setSuccess(false);
        try {
            String taskTimeFrom = schedule.getTaskTimeFrom();
            String taskTimeEnd = schedule.getTaskTimeEnd();
            Long practitionerId = schedule.getPractitionerId();
            // 判断活动时间段是否冲突
            if (!CommonUtil.isNullOrBlank(taskTimeFrom) && !CommonUtil.isNullOrBlank(taskTimeEnd)) {
                int count = scheduleTrackMapper.checkTimePeriodConflict(taskTimeFrom, taskTimeEnd, practitionerId);
                if (count > 0) {
                    result.setMessage(ZHBErrorConfig.getErrorInfo("830020"));
                    return result;
                }
            } else {
                result.setMessage("taskTimeFrom,taskTimeEnd不能为空");
                return result;
            }

            schedule.setCreatedBy(practitionerId);
            schedule.setUpdatedBy(practitionerId);
            schedule.setUpdatorType(2);
            schedule.setCreatorType(2);
            Date curTime = new Date();
            schedule.setUpdatedAt(new Date());
            schedule.setCreatedAt(curTime);
            scheduleTrackMapper.insert(schedule);
            // 营销日程同步插入到商机活动跟踪表，增员日程同步插入到增援活动跟踪表
            Integer taskType = schedule.getTaskType();
            Long referLeadsId = schedule.getReferLeadsId();
            Long referPotentialId = schedule.getReferPotentialId();
            if (1 == taskType && null != referLeadsId) {
                this.insertOpportunityRecord(schedule);
            }
            if (2 == taskType && null != referPotentialId) {
                this.insertPotentialAssignedTrack(schedule);
            }
            result.setSuccess(true);
            result.setMessage(ZHBErrorConfig.getErrorInfo("800000"));
        } catch (Exception e) {
            e.printStackTrace();
            result.setMessage(e.getMessage());
        }
        return result;
    }

    private void insertOpportunityRecord(ScheduleTrack schedule) {
        MktLeadsAssignedTrack track = new MktLeadsAssignedTrack();
        CommonUtil.simpleObjectCopy(schedule, track);

        track.setLeadsAssignedId(schedule.getReferLeadsId());
        track.setSalesNotice(schedule.getNotice());
        List<MktLeadsAssignedTrack> trackList = mktLeadsAssignedTrackDALService.findByTrackTimeForNew(track);
        if (!trackList.isEmpty()){
            //保存
            mktLeadsAssignedTrackDALService.saveTrack(track);
        }

    }

    private void insertPotentialAssignedTrack(ScheduleTrack schedule) {
        Long practitionerId = schedule.getPractitionerId();
        Long potentialId = schedule.getReferPotentialId();
        Long trackStatusId = schedule.getMdDropOptionId();
        String notice = schedule.getNotice();
        Date trackTimeDate = schedule.getTrackTime();

        AclPractitionerPotentialAssignedTrack assignedTrack = new AclPractitionerPotentialAssignedTrack();

        //2、查询团队长对该增员所有增员状态
        List<AclPractitionerPotentialAssignedTrack> assignedTrackList = aclPractitionerPotentialAssignedTrackDALService.findByPractitionerIdAndPotentialIdAndTrackStatusId(practitionerId,potentialId,trackStatusId,trackTimeDate);
        if (assignedTrackList == null || assignedTrackList.isEmpty()) {//3、如果没有，获取增员状态对应的分值，
            MdDropOptions options = mdDropOptionsDALService.findById(trackStatusId);
            assignedTrack.setTrackScore(options.getDropOptionScore());
            assignedTrack.setPractitionerAssignedId(practitionerId);
            assignedTrack.setPractitionerPotentialId(potentialId);
            assignedTrack.setTrackStatusId(trackStatusId);
            assignedTrack.setNotice(notice);
            assignedTrack.setTrackTime(trackTimeDate);
            assignedTrack.setIsActive(1);
            assignedTrack.setIsLasted(1);
            assignedTrack.setCreatorType(2);
            assignedTrack.setCreatedBy(practitionerId);
            assignedTrack.setCreatedAt(new Date());
            aclPractitionerPotentialAssignedTrackDALService.save(assignedTrack);

        } else {
            assignedTrack = assignedTrackList.get(0);
            assignedTrack.setNotice(notice);
            assignedTrack.setIsLasted(1);
            assignedTrack.setUpdatedAt(new Date());
            assignedTrack.setUpdatedBy(practitionerId);
            assignedTrack.setUpdaterType(2);
            aclPractitionerPotentialAssignedTrackDALService.update(assignedTrack);
        }

    }

    @Override
    public JsonResult queryScheduleTrackList(Long practitionerId, String trackDate) {
        JsonResult result = new JsonResult();
        result.setSuccess(false);
        if (StringUtils.isEmpty(trackDate) || null == practitionerId) {
            result.setMessage("经纪人id和查询时间不能为空");
            return result;
        }
        List<ScheduleTrack> showList = new ArrayList<>();
        try {
            List<ScheduleTrack> dataList = scheduleTrackMapper.queryScheduleTrackList(practitionerId);
            for (ScheduleTrack schedule: dataList) {
                if (this.checkFixedDay(trackDate, schedule)) {
                    showList.add(schedule);
                }
            }
            result.setData(showList);
            result.setSuccess(true);
            result.setMessage(ZHBErrorConfig.getErrorInfo("800000"));
        } catch (Exception e) {
            e.printStackTrace();
            result.setMessage(e.getMessage());
        }
        return result;
    }

    private boolean checkFixedDay(String trackDate, ScheduleTrack track) {
        Date date = CommonUtil.stringParseDate(trackDate, "yyyy-MM-dd");
        String week = CommonUtil.getWeekOfDate(date);
        String trackTime = CommonUtil.dateParseString(track.getTrackTime(), "yyyy-MM-dd");
        int flag = 0;
        switch (week) {
            case "1":
                flag = track.getTaskRoutineAtweek1();
                break;
            case "2":
                flag = track.getTaskRoutineAtweek2();
                break;
            case "3":
                flag = track.getTaskRoutineAtweek3();
                break;
            case "4":
                flag = track.getTaskRoutineAtweek4();
                break;
            case "5":
                flag = track.getTaskRoutineAtweek5();
                break;
            case "6":
                flag = track.getTaskRoutineAtweek6();
                break;
            case "0":
                flag = track.getTaskRoutineAtweek7();
                break;
            default:
                break;
        }
        if (CommonUtil.isToday(trackTime) || flag == 1) {
            return true;
        } else {
            return false;
        }
    }


}
