package com.yd.api.practitioner.service.impl;

import com.yd.api.practitioner.service.ScheduleTrackService;
import com.yd.api.practitioner.vo.sechedule.*;
import com.yd.api.result.CommonResult;
import com.yd.dal.entity.customer.AclPractitionerPotentialAssignedTrack;
import com.yd.dal.entity.marketing.MktLeadsAssignedTrack;
import com.yd.dal.entity.meta.MdDropOptions;
import com.yd.dal.entity.practitioner.PersonalSchedule;
import com.yd.dal.entity.practitioner.ScheduleTrack;
import com.yd.dal.mapper.marketing.ScheduleTrackMapper;
import com.yd.dal.service.customer.AclPractitionerPotentialAssignedTrackDALService;
import com.yd.dal.service.marketing.MktLeadsAssignedTrackDALService;
import com.yd.dal.service.meta.MdDropOptionsDALService;
import com.yd.util.CommonUtil;
import com.yd.util.config.ZHBErrorConfig;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.*;

@Service
public class ScheduleTrackServiceImpl implements ScheduleTrackService {
    @Autowired
    private ScheduleTrackMapper scheduleTrackMapper;
    @Autowired
    private AclPractitionerPotentialAssignedTrackDALService aclPractitionerPotentialAssignedTrackDALService;
    @Autowired
    private MdDropOptionsDALService mdDropOptionsDALService;
    @Autowired
    private MktLeadsAssignedTrackDALService mktLeadsAssignedTrackDALService;

    @Override
    public AddScheduleTrackResponseVO insert(AddScheduleTrackRequestVO requestVO) {
        // 响应实体
        AddScheduleTrackResponseVO resp = new AddScheduleTrackResponseVO();
        // 数据库实体
        ScheduleTrack schedule = new ScheduleTrack();
        BeanUtils.copyProperties(requestVO, schedule);
        try {
            String taskTimeFrom = schedule.getTaskTimeFrom();
            String taskTimeEnd = schedule.getTaskTimeEnd();
            Long practitionerId = schedule.getPractitionerId();
            Long mdDropOptionId = schedule.getMdDropOptionId();
            // 判断活动时间段是否冲突
            if (!CommonUtil.isNullOrBlank(taskTimeFrom) && !CommonUtil.isNullOrBlank(taskTimeEnd)) {
                int count = scheduleTrackMapper.checkTimePeriodConflict(taskTimeFrom, taskTimeEnd, practitionerId);
                if (count > 0) {
                    resp.setCommonResult(new CommonResult(true, ZHBErrorConfig.getErrorInfo("830020")));
                    return resp;
                }
            } else {
                resp.setCommonResult(new CommonResult(true, "taskTimeFrom,taskTimeEnd不能为空"));
                return resp;
            }

            Map<Long, MdDropOptions> taskTypeMap = this.getScheduleTaskTypeMap();
            Integer trackScore = taskTypeMap.get(mdDropOptionId).getDropOptionScore();
            schedule.setTrackScore(trackScore);
            schedule.setCreatedBy(practitionerId);
            schedule.setUpdatedBy(practitionerId);
            schedule.setUpdatorType(2);
            schedule.setCreatorType(2);
            Date curTime = new Date();
            schedule.setUpdatedAt(new Date());
            schedule.setCreatedAt(curTime);
            scheduleTrackMapper.insert(schedule);
            // 营销日程同步插入到商机活动跟踪表，增员日程同步插入到增援活动跟踪表
            Integer taskType = schedule.getTaskType();
            Long referLeadsId = schedule.getReferLeadsId();
            Long referPotentialId = schedule.getReferPotentialId();
            if (1 == taskType && null != referLeadsId) {
                this.insertOpportunityRecord(schedule);
            }
            if (2 == taskType && null != referPotentialId) {
                this.insertPotentialAssignedTrack(schedule);
            }
            resp.setCommonResult(new CommonResult(true, ZHBErrorConfig.getErrorInfo("800000")));
        } catch (Exception e) {
            e.printStackTrace();
            resp.setCommonResult(new CommonResult(false, ZHBErrorConfig.getErrorInfo("800001")));
        }
        return resp;
    }

    private void insertOpportunityRecord(ScheduleTrack schedule) {
        MktLeadsAssignedTrack track = new MktLeadsAssignedTrack();
        CommonUtil.simpleObjectCopy(schedule, track);

        track.setLeadsAssignedId(schedule.getReferLeadsId());
        track.setSalesNotice(schedule.getNotice());
        List<MktLeadsAssignedTrack> trackList = mktLeadsAssignedTrackDALService.findByTrackTimeForNew(track);
        if (!trackList.isEmpty()){
            //保存
            mktLeadsAssignedTrackDALService.saveTrack(track);
        }

    }

    private void insertPotentialAssignedTrack(ScheduleTrack schedule) {
        Long practitionerId = schedule.getPractitionerId();
        Long potentialId = schedule.getReferPotentialId();
        Long trackStatusId = schedule.getMdDropOptionId();
        String notice = schedule.getNotice();
        Date trackTimeDate = schedule.getTrackTime();

        AclPractitionerPotentialAssignedTrack assignedTrack = new AclPractitionerPotentialAssignedTrack();

        //2、查询团队长对该增员所有增员状态
        List<AclPractitionerPotentialAssignedTrack> assignedTrackList = aclPractitionerPotentialAssignedTrackDALService.findByPractitionerIdAndPotentialIdAndTrackStatusId(practitionerId,potentialId,trackStatusId,trackTimeDate);
        if (assignedTrackList == null || assignedTrackList.isEmpty()) {//3、如果没有，获取增员状态对应的分值，
            MdDropOptions options = mdDropOptionsDALService.findById(trackStatusId);
            assignedTrack.setTrackScore(options.getDropOptionScore());
            assignedTrack.setPractitionerAssignedId(practitionerId);
            assignedTrack.setPractitionerPotentialId(potentialId);
            assignedTrack.setTrackStatusId(trackStatusId);
            assignedTrack.setNotice(notice);
            assignedTrack.setTrackTime(trackTimeDate);
            assignedTrack.setIsActive(1);
            assignedTrack.setIsLasted(1);
            assignedTrack.setCreatorType(2);
            assignedTrack.setCreatedBy(practitionerId);
            assignedTrack.setCreatedAt(new Date());
            aclPractitionerPotentialAssignedTrackDALService.save(assignedTrack);

        } else {
            assignedTrack = assignedTrackList.get(0);
            assignedTrack.setNotice(notice);
            assignedTrack.setIsLasted(1);
            assignedTrack.setUpdatedAt(new Date());
            assignedTrack.setUpdatedBy(practitionerId);
            assignedTrack.setUpdaterType(2);
            aclPractitionerPotentialAssignedTrackDALService.update(assignedTrack);
        }

    }

    @Override
    public QueryScheduleTrackListResponseVO queryScheduleTrackList(QueryScheduleTrackListRequestVO requestVO) {
        QueryScheduleTrackListResponseVO resp = new QueryScheduleTrackListResponseVO();

        Long practitionerId = requestVO.getPractitionerId();
        String trackDate = requestVO.getTrackDate();
        if (StringUtils.isEmpty(trackDate) || null == practitionerId) {
            resp.setCommonResult(new CommonResult(true, "经纪人id和查询时间不能为空"));
            return resp;
        }
        List<ScheduleTrackVO> showList = new ArrayList<>();
        try {
            List<ScheduleTrack> dataList = scheduleTrackMapper.queryScheduleTrackList(practitionerId);
            Map<Long, MdDropOptions> taskTypeMap = this.getScheduleTaskTypeMap();
            String optionName = null;
            for (ScheduleTrack schedule: dataList) {
                if (this.checkFixedDay(trackDate, schedule)) {
                    ScheduleTrackVO scheduleTrackVO = new ScheduleTrackVO();
                    BeanUtils.copyProperties(schedule, scheduleTrackVO);
                    optionName = taskTypeMap.get(schedule.getMdDropOptionId()).getDropOptionName();
                    scheduleTrackVO.setMdDropOptionName(optionName);
                    showList.add(scheduleTrackVO);
                }
            }
            resp.setDataList(showList);
            resp.setCommonResult(new CommonResult(true, ZHBErrorConfig.getErrorInfo("800000")));
        } catch (Exception e) {
            e.printStackTrace();
            resp.setCommonResult(new CommonResult(true, ZHBErrorConfig.getErrorInfo("800001")));
        }
        return resp;
    }

    private Map<Long, MdDropOptions> getScheduleTaskTypeMap() {
        Map<Long, MdDropOptions> saleOptMap = mdDropOptionsDALService.queryIdObjMapByByDropMasterCode("bizchance_promotion_action");
        Map<Long, MdDropOptions> recruitOptMap = mdDropOptionsDALService.queryIdObjMapByByDropMasterCode("team_building_track");
        Map<Long, MdDropOptions> otherOptMap = mdDropOptionsDALService.queryIdObjMapByByDropMasterCode("pep_schedule_task_others");

        Map<Long, MdDropOptions> taskTypeMap = new HashMap<>();
        taskTypeMap.putAll(saleOptMap);
        taskTypeMap.putAll(recruitOptMap);
        taskTypeMap.putAll(otherOptMap);
        return taskTypeMap;
    }

    private boolean checkFixedDay(String trackDate, ScheduleTrack track) {
        Date date = CommonUtil.stringParseDate(trackDate, "yyyy-MM-dd");
        String week = CommonUtil.getWeekOfDate(date);
        String trackTime = CommonUtil.dateParseString(track.getTrackTime(), "yyyy-MM-dd");
        int flag = 0;
        switch (week) {
            case "1":
                flag = track.getTaskRoutineAtweek1();
                break;
            case "2":
                flag = track.getTaskRoutineAtweek2();
                break;
            case "3":
                flag = track.getTaskRoutineAtweek3();
                break;
            case "4":
                flag = track.getTaskRoutineAtweek4();
                break;
            case "5":
                flag = track.getTaskRoutineAtweek5();
                break;
            case "6":
                flag = track.getTaskRoutineAtweek6();
                break;
            case "0":
                flag = track.getTaskRoutineAtweek7();
                break;
            default:
                break;
        }
        if (trackTime.equals(trackDate) || flag == 1) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public QueryPEPScoreResponseVO queryPEPScore(Long practitionerId) {
        QueryPEPScoreResponseVO resp = new QueryPEPScoreResponseVO();
        // 获取本月第一天
        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.MONTH,0);
        // 获取到本月起始日
        int actualMinimum = cal.getActualMinimum(Calendar.DAY_OF_MONTH);
        cal.set(cal.get(Calendar.YEAR), cal.get(Calendar.MONDAY), actualMinimum,00,00,00);
        // 获取本月起始日的年月日时分秒格式
        Date monthOne = cal.getTime();

        // 获取本月当前日的年月日时分秒格式
        Date today = new Date();
        // 获取本周第一天（周一为始）
        cal.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        Date weekOne = cal.getTime();

        // 根据团队长经纪人id查询团队每个成员本日，本周，本月得分以及本月FYC
        List<Map<String, Object>> saleAndRecuit = scheduleTrackMapper.querySaleRecuitScoreFYC(practitionerId, monthOne, today);

        // 其他活动有长期固定，通过计算时间段内展示次数，算得总分
        // 先获得每个人所有的其他活动
        List<Long> practitionerIdList = scheduleTrackMapper.queryTeamAllPractitionerId(practitionerId);
        List<PersonalSchedule> personalScheduleList = scheduleTrackMapper.queryPersonalScheduleList(practitionerIdList);

        // 根据活动算本日，本周，本月的分数
        this.computeOtherScore(personalScheduleList, today, weekOne, monthOne);
        // 最后合并营销增员和其他活动的分数
        this.mergeScore(personalScheduleList, saleAndRecuit);

        resp.setPersonalList(saleAndRecuit);

        System.out.println("看看结果集");
        System.out.println(saleAndRecuit);
        return resp;
    }

    private void mergeScore(List<PersonalSchedule> personalScheduleList, List<Map<String, Object>> saleAndRecuit) {
        DecimalFormat df = new DecimalFormat("#.##%");
        for (Map<String, Object> scoreMap:saleAndRecuit) {

            for (PersonalSchedule perSchedule:personalScheduleList) {
                Long curPractitionerId = perSchedule.getPractitionerId();
                if (curPractitionerId.equals(scoreMap.get("practitionerId"))) {
                    Map<String, Long> otherMap = perSchedule.getOther();

                    Long dayScore = Long.parseLong(scoreMap.get("dayScore").toString()) + otherMap.get("dayScore");
                    Long weekScore = Long.parseLong(scoreMap.get("weekScore").toString()) + otherMap.get("weekScore");
                    Long monthScore = Long.parseLong(scoreMap.get("monthScore").toString()) + otherMap.get("monthScore");

                    scoreMap.put("dayScore", dayScore);
                    scoreMap.put("weekScore", weekScore);
                    scoreMap.put("monthScore", monthScore);

                }

            }
            // 单人分效 scoreMap.get("monthScore").toString()
            Double rate = BigDecimal.valueOf(Double.parseDouble(scoreMap.get("FYC").toString()))
                    .divide(BigDecimal.valueOf(Long.parseLong(scoreMap.get("monthScore").toString()) ), 2, BigDecimal.ROUND_HALF_UP)
                    .doubleValue();
            scoreMap.put("rate", df.format(rate));
        }

    }

    private void computeOtherScore(List<PersonalSchedule> personalScheduleList,  Date today, Date weekOne, Date monthOne) {
        Integer trackScore = null;
        long dayOther = 0L;
        long weekOther = 0L;
        long monthOther = 0L;
        for (PersonalSchedule perSchedule:personalScheduleList) {
            List<ScheduleTrack> scheduleList = perSchedule.getScheduleList();
            // 计算所有其他活动的本日，本周，本月得分
            for (ScheduleTrack s:scheduleList) {
                trackScore = s.getTrackScore();

                dayOther += this.calculateScheduleDisplayCount(s, today, today) * trackScore;
                weekOther += this.calculateScheduleDisplayCount(s, weekOne, today) * trackScore;
                monthOther += this.calculateScheduleDisplayCount(s, monthOne, today) * trackScore;

                Map<String, Long> map = new HashMap<>();
                map.put("dayScore", dayOther);
                map.put("weekScore", weekOther);
                map.put("monthScore", monthOther);

                perSchedule.setOther(map);
            }
        }
    }

    private long calculateScheduleDisplayCount(ScheduleTrack s, Date firstDay, Date lastDay) {
        long count = 0L;
        if (1 == s.getTaskRoutineAtweek1()) {
            count += CommonUtil.weekDayCount(firstDay, lastDay, 1);
        }
        if (1 == s.getTaskRoutineAtweek2()) {
            count += CommonUtil.weekDayCount(firstDay, lastDay, 2);
        }
        if (1 == s.getTaskRoutineAtweek3()) {
            count += CommonUtil.weekDayCount(firstDay, lastDay, 3);
        }
        if (1 == s.getTaskRoutineAtweek4()) {
            count += CommonUtil.weekDayCount(firstDay, lastDay, 4);
        }
        if (1 == s.getTaskRoutineAtweek5()) {
            count += CommonUtil.weekDayCount(firstDay, lastDay, 5);
        }
        if (1 == s.getTaskRoutineAtweek6()) {
            count += CommonUtil.weekDayCount(firstDay, lastDay, 6);
        }
        if (1 == s.getTaskRoutineAtweek7()) {
            count += CommonUtil.weekDayCount(firstDay, lastDay, 7);
        }
        return count;
    }

}
