import { ref, reactive, computed } from 'vue'
import { ElMessage, ElNotification, ElLoading } from 'element-plus'
import { UploadFile, UploadRawFile } from 'element-plus/es/components/upload/src/upload'
import axios, { AxiosRequestConfig, AxiosProgressEvent } from 'axios'

// 上传配置接口定义
export interface UploadConfig {
  // 上传接口URL
  url: string
  // 后端接收文件的字段名
  fieldName?: string
  // 最大文件大小(MB)
  maxSize?: number
  // 允许的文件类型，例如['image/jpeg', 'image/png']
  allowedTypes?: string[]
  // 最大上传文件数量
  maxCount?: number
  // 额外的请求参数
  extraParams?: Record<string, any>
  // 上传请求头
  headers?: Record<string, string>
  // 是否支持多文件上传
  multiple?: boolean
}

// 上传进度信息接口
export interface UploadProgress {
  percent: number
  uploaded: number
  total: number
  file: UploadRawFile
}

// 上传结果接口
export interface UploadResult {
  success?: boolean
  data?: unknown
  msg?: string
  code?: number
  error?: {
    message: string
    code?: number
  }
  file: UploadRawFile
}

// 上传文件信息接口
export interface UploadFileInfo extends UploadFile {
  progress: number
  uploadId?: string
}

// 默认配置
const defaultConfig: UploadConfig = {
  url: `${import.meta.env.VITE_REMOTE_API_BASE_URL}/oss/api/oss/upload`,
  fieldName: 'file',
  maxSize: 10,
  allowedTypes: [],
  maxCount: 10,
  extraParams: {},
  headers: {},
  multiple: false,
}

/**
 * 文件上传组合式API
 * @param config 上传配置
 * @returns 上传相关方法和状态
 */
export function useFileUpload(config: Partial<UploadConfig> = {}) {
  // 合并配置
  const uploadConfig = { ...defaultConfig, ...config }

  // 上传文件列表
  const fileList = reactive<UploadFileInfo[]>([])
  // 上传中状态
  const isUploading = ref(false)
  // 上传进度
  const uploadProgress = ref<UploadProgress | null>(null)
  // 上传ID生成器
  let uploadIdCounter = 0

  // 计算已上传文件数量
  const uploadedCount = computed(() => {
    return fileList.filter((file) => file.status === 'success').length
  })

  // 计算上传失败文件数量
  const errorCount = computed(() => {
    return fileList.filter((file) => file.status === 'error').length
  })

  // 生成唯一上传ID
  const generateUploadId = () => {
    uploadIdCounter++
    return `upload_${Date.now()}_${uploadIdCounter}`
  }

  /**
   * 验证文件是否符合要求
   * @param file 待验证文件
   * @returns 验证结果和错误信息
   */
  const validateFile = (file: UploadRawFile): { valid: boolean; message?: string } => {
    // 验证文件大小
    if (uploadConfig.maxSize) {
      const maxSizeBytes = uploadConfig.maxSize * 1024 * 1024
      if (file.size > maxSizeBytes) {
        return {
          valid: false,
          message: `文件大小不能超过${uploadConfig.maxSize}MB`,
        }
      }
    }

    // 验证文件类型
    if (uploadConfig.allowedTypes && uploadConfig.allowedTypes.length > 0) {
      // 检查MIME类型
      const mimeTypeMatch = uploadConfig.allowedTypes.includes(file.type)
      // 检查文件扩展名
      const ext = file.name.split('.').pop()?.toLowerCase()
      const extMatch = ext ? uploadConfig.allowedTypes.some((type) => type.includes(ext)) : false

      if (!mimeTypeMatch && !extMatch) {
        return {
          valid: false,
          message: `不支持的文件类型，允许的类型: ${uploadConfig.allowedTypes.join(', ')}`,
        }
      }
    }

    // 验证文件数量
    if (uploadConfig.maxCount && fileList.length >= uploadConfig.maxCount) {
      return {
        valid: false,
        message: `最多只能上传${uploadConfig.maxCount}个文件`,
      }
    }

    return { valid: true }
  }

  /**
   * 处理文件选择
   * @param files 选中的文件列表
   */
  const handleFileSelect = (files: UploadRawFile[]) => {
    if (!files || files.length === 0) return

    files.forEach((file) => {
      const validation = validateFile(file)
      if (validation.valid) {
        // 添加到文件列表
        fileList.push({
          uid: file.uid,
          name: file.name,
          raw: file,
          size: file.size,
          status: 'ready',
          progress: 0,
        })
      } else if (validation.message) {
        ElMessage.error(validation.message)
      }
    })
  }

  /**
   * 上传单个文件
   * @param file 文件信息
   * @returns 上传结果
   */
  const uploadSingleFile = async (file: UploadFileInfo): Promise<UploadResult> => {
    return new Promise((resolve) => {
      if (!file.raw) {
        const result: UploadResult = {
          success: false,
          error: { message: '文件不存在' },
          file: file.raw as UploadRawFile,
        }
        return resolve(result)
      }

      // 创建FormData
      const formData = new FormData()
      formData.append(uploadConfig.fieldName as string, file.raw, file.name)

      // 添加额外参数
      if (uploadConfig.extraParams) {
        Object.entries(uploadConfig.extraParams).forEach(([key, value]) => {
          formData.append(key, value)
        })
      }

      // 更新文件状态
      file.status = 'uploading'
      file.uploadId = generateUploadId()
      isUploading.value = true

      // 配置axios请求
      const axiosConfig: AxiosRequestConfig = {
        headers: {
          'Content-Type': 'multipart/form-data',
          Authorization: `${localStorage.getItem('authToken') || ''}`,
          ...uploadConfig.headers,
        },
        onUploadProgress: (progressEvent: AxiosProgressEvent) => {
          if (progressEvent.total) {
            const percent = Math.round((progressEvent.loaded / progressEvent.total) * 100)
            file.progress = percent

            uploadProgress.value = {
              percent,
              uploaded: progressEvent.loaded,
              total: progressEvent.total,
              file: file.raw as UploadRawFile,
            }
          }
        },
      }

      // 发送上传请求
      axios
        .post(uploadConfig.url, formData, axiosConfig)
        .then((response) => {
          file.status = 'success'
          ElMessage.success(`文件 "${file.name}" 上传成功`)

          resolve({
            success: true,
            data: response.data,
            file: file.raw as UploadRawFile,
          })
        })
        .catch((error) => {
          file.status = 'error'
          const errorMsg = error.response?.data?.message || `文件 "${file.name}" 上传失败`
          ElMessage.error(errorMsg)

          resolve({
            success: false,
            error: {
              message: errorMsg,
              code: error.response?.status,
            },
            file: file.raw as UploadRawFile,
          })
        })
        .finally(() => {
          // 检查是否还有上传中的文件
          const hasUploading = fileList.some((f) => f.status === 'uploading')
          if (!hasUploading) {
            isUploading.value = false
            uploadProgress.value = null
          }
        })
    })
  }

  /**
   * 上传所有待上传文件
   * @returns 所有文件的上传结果
   */
  const uploadAllFiles = async (): Promise<UploadResult[]> => {
    const readyFiles = fileList.filter((file) => file.status === 'ready')

    if (readyFiles.length === 0) {
      ElMessage.warning('没有待上传的文件')
      return []
    }

    const loading = ElLoading.service({
      lock: true,
      text: '正在上传文件...',
      background: 'rgba(0, 0, 0, 0.7)',
    })

    try {
      // 依次上传文件
      const results: UploadResult[] = []
      for (const file of readyFiles) {
        const result = await uploadSingleFile(file)
        results.push(result)
      }

      ElNotification.success({
        title: '上传完成',
        message: `成功上传 ${results.filter((r) => r.success).length}/${results.length} 个文件`,
        duration: 3000,
      })

      return results
    } finally {
      loading.close()
    }
  }

  /**
   * 移除文件
   * @param file 要移除的文件
   */
  const removeFile = (file: UploadFileInfo) => {
    const index = fileList.indexOf(file)
    if (index !== -1) {
      // 如果是上传中的文件，先取消上传
      if (file.status === 'uploading' && file.uploadId) {
        // 这里可以实现取消上传的逻辑
        ElMessage.info(`已取消 "${file.name}" 的上传`)
      }
      fileList.splice(index, 1)
    }
  }

  /**
   * 清空文件列表
   */
  const clearFiles = () => {
    if (isUploading.value) {
      ElMessage.warning('有文件正在上传中，不能清空列表')
      return
    }
    fileList.length = 0
  }

  return {
    fileList,
    isUploading,
    uploadProgress,
    uploadedCount,
    errorCount,
    handleFileSelect,
    uploadSingleFile,
    uploadAllFiles,
    removeFile,
    clearFiles,
    validateFile,
  }
}
