import axios, { type AxiosError, type AxiosResponse, type InternalAxiosRequestConfig } from 'axios'

// 定义统一的响应类型
export interface ApiResponse<T = object> {
  code: number
  message?: string
  data: T
  success: boolean
  msg?: string
}

// 创建axios实例
const request = axios.create({
  baseURL: '/',
  timeout: 10000,
  headers: {
    'Content-Type': 'application/json',
    Authorization: localStorage.getItem('authToken') || '',
  },
})

// 请求拦截器 - 添加Authorization头
request.interceptors.request.use(
  (config: InternalAxiosRequestConfig) => {
    // 从本地存储获取token
    const token = localStorage.getItem('authToken')

    // 如果token存在，添加到请求头
    if (token && config.headers) {
      config.headers.Authorization = token
    }

    return config
  },
  (error: AxiosError) => {
    // 处理请求错误
    return Promise.reject(error)
  },
)

// 响应拦截器 - 处理常见错误
request.interceptors.response.use(
  (response: AxiosResponse) => {
    // 直接返回响应数据
    return response.data
  },
  (error: AxiosError) => {
    // 处理401未授权错误
    if (error.response && error.response.status === 401) {
      // 清除无效token
      localStorage.removeItem('authToken')
      // 如果不是登录页面，跳转到登录页
      if (!window.location.pathname.includes('/login')) {
        // 保存当前URL，登录后可跳转回来
        localStorage.setItem('redirectPath', window.location.pathname)
        window.location.href = '/yd-email/login'
      }
    }

    return Promise.reject(error)
  },
)

export default request
