import { Injectable } from '@angular/core';
import { DatePipe } from '@angular/common';
import { Meta, Title } from '@angular/platform-browser';
import { Subject } from "rxjs/index";
declare var wx: any;
@Injectable({
  providedIn: 'root'
})
export class LifeCommonService {
  private shareStatusSource = new Subject<string>();
  shareStatus$ = this.shareStatusSource.asObservable();

  constructor(private datePipe: DatePipe,
    private titleService: Title,
    private metaService: Meta) {
  }

  // 向订阅者发布分享结果
  // 0.关闭分享引导页
  // 1.用户点击分享
  // 2.已经分享过了
  shareStatusPublish(type) {
    this.shareStatusSource.next(type);
  }

  /**
   * 检测操作系统
   */
  checkOs() {
    let os;
    const isWin = (navigator.platform === 'Win32') || (navigator.platform === 'Windows');
    const isMac = (navigator.platform === 'Mac68K') ||
      (navigator.platform === 'MacPPC') ||
      (navigator.platform === 'Macintosh') ||
      (navigator.platform === 'MacIntel');
    if (/(iPhone|iPad|iPod|iOS)/i.test(navigator.userAgent)) {
      os = 1;
      return os; // 这是iOS平台下浏览器
    } else if (/android/i.test(navigator.userAgent)) {
      os = 2;
      return os; // 这是Android平台下浏览器
    } else if (isWin) {
      os = 3;
      return os; // 这是Android平台下浏览器
    } else if (isMac) {
      os = 4;
      return os; // 这是mac系统
    } else if (/Linux/i.test(navigator.userAgent)) {
      os = 5;
      return os; // 这是Linux平台下浏览器
    } else {
      os = 6;
      return os;
    }
  }

  /**
   * 检测设备信息
   * deviceType:PC为1,移动端为2,微信为3
   */
  checkDeviceType() {
    let deviceType = null;
    if (/Android|webOS|iPhone|iPod|BlackBerry/i.test(navigator.userAgent)) {
      if (navigator.userAgent.toLowerCase().indexOf('micromessenger') !== -1) {
        deviceType = '3';
      } else {
        deviceType = '2';
      }
    } else {
      deviceType = '1';
    }
    return deviceType;
  }

  /**
   * shareCode生成器
   */
  generateShareCode(jsonParams?: any) {
    const shareCode = {
      shareType: '1',
      os: this.checkOs(),
      channel: '0',
      isOrder: 0,
      jsonParams: JSON.stringify(jsonParams)
    };
    return shareCode;
  }

  // 设置title
  setTitle(title) {
    this.titleService.setTitle(title);
  }

  // 更新meta标签
  updateMeta(keywords, description) {
    this.metaService.updateTag({ content: keywords }, 'name=keywords');
    this.metaService.updateTag({ content: description }, 'name=description');
  }

  wxShare(title: string, desc: string, link: string, imgUrl: string) {
    const shareData = {
      title: title, // 分享标题
      desc: desc, // 分享描述
      link: link, // 分享链接，该链接域名或路径必须与当前页面对应的公众号JS安全域名一致
      imgUrl: imgUrl, // 分享图标
      success: () => {
        this.shareStatusPublish(0);
      }
    };
    wx.ready(() => {   // 需在用户可能点击分享按钮前就先调用
      wx.updateAppMessageShareData(shareData);
      wx.updateTimelineShareData(shareData);
    });
  }

  // 根据开始时间设置结束时间,天数应减1，当日~当日是一天
  setInitDate(dayCount) {
    let dt;
    let dtEnd;
    (dt = new Date()).setDate(dt.getDate() + 1);
    (dtEnd = new Date(dt)).setDate(dt.getDate() + dayCount * 1 - 1);
    const initDate = {
      startDate: Date.parse(dt.toDateString()),
      endDate: Date.parse(dtEnd.toDateString())
    };
    sessionStorage.removeItem('timeRange');
    sessionStorage.setItem('initDate', JSON.stringify(initDate));
    const date = {
      startDate: dt,
      endDate: dtEnd
    };
    return date;
  }

  // 判断一年是365还是366
  yearDayCount(days, disableCalc?: boolean) {
    if (days === 365 || days === 366) {
      const y = new Date().getFullYear();
      const isLeap = (0 === y % 4) && (0 !== y % 100) || (0 === y % 400);
      days = isLeap ? 366 : 365;
      if (disableCalc) {
        days = 365;
      }
    }
    return days;
  }

  dateFormat(date: any, format?: string) {
    return this.datePipe.transform(date, format);
  }

  delHtmlTag(str) {
    return str.replace(/<[^>]+>/g, "");//去掉所有的html标记
  }
}
