package com.yd.base.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.yd.base.api.service.ApiRelFieldValueService;
import com.yd.base.feign.request.relfieldvalue.ApiRelFieldValuePageRequest;
import com.yd.base.feign.request.relfieldvalue.GetFieldListRequest;
import com.yd.base.feign.response.relfieldvalue.ApiRelFieldValuePageResponse;
import com.yd.base.feign.response.relfieldvalue.ApiRelFieldValueResponse;
import com.yd.base.feign.response.relfieldvalue.GetFieldListResponse;
import com.yd.base.feign.response.relfieldvalue.GetFieldValueListResponse;
import com.yd.base.service.dto.FieldDto;
import com.yd.base.service.dto.RelFieldValueDto;
import com.yd.base.service.model.Field;
import com.yd.base.service.model.FvqSetting;
import com.yd.base.service.model.RelFieldValue;
import com.yd.base.service.service.IFieldService;
import com.yd.base.service.service.IFvqSettingService;
import com.yd.base.service.service.IRelFieldValueService;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.TokenUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiRelFieldValueServiceImpl implements ApiRelFieldValueService {

    @Autowired
    private IRelFieldValueService iRelFieldValueService;

    @Autowired
    private IFvqSettingService iFvqSettingService;

    @Autowired
    private IFieldService iFieldService;

    @Autowired
    private RestTemplate restTemplate;

    @Autowired
    private ObjectMapper objectMapper;

    /**
     * 保存字段值和字段的关系
     * @param fieldValueBizId
     * @param fieldBizIdList
     * @return
     */
    @Override
    public Result save(String fieldValueBizId, List<String> fieldBizIdList) {
        //先删后新增
        //根据字段值业务ID删除关系
        iRelFieldValueService.delByFieldValueBizId(fieldValueBizId);

        //新增
        if (!CollectionUtils.isEmpty(fieldBizIdList)) {
            List<RelFieldValue> relFieldValueList = fieldBizIdList.stream().map(fieldBizId -> {
                RelFieldValue relFieldValue = new RelFieldValue();
                relFieldValue.setFieldBizId(fieldBizId);
                relFieldValue.setFieldValueBizId(fieldValueBizId);
                return relFieldValue;
            }).collect(Collectors.toList());

            iRelFieldValueService.saveOrUpdateBatch(relFieldValueList);
        }
        return Result.success();
    }

    /**
     * 分页列表查询-字段和字段值关系信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiRelFieldValuePageResponse>> page(ApiRelFieldValuePageRequest request) {
        Page<ApiRelFieldValuePageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiRelFieldValuePageResponse> iPage = iRelFieldValueService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 根据字段ID列表获取字段值列表
     * @param request
     * @return
     */
    @Override
    public Result<List<GetFieldListResponse>> getFieldList(GetFieldListRequest request) {
        List<GetFieldListResponse> resultList = new ArrayList<>();

        // 检查字段ID列表是否为空
        if (CollectionUtils.isEmpty(request.getFieldBizIdList())) {
            return Result.success(resultList);
        }

        // 1. 根据字段ID列表查询fvq_setting配置
        Map<String, FvqSetting> fvqSettingMap = getFvqSettingsByFieldIds(request.getFieldBizIdList());

        // 2. 分离有配置和无配置的字段ID
        List<String> configuredFieldIds = new ArrayList<>();
        List<String> unconfiguredFieldIds = new ArrayList<>();

        for (String fieldBizId : request.getFieldBizIdList()) {
            if (fvqSettingMap.containsKey(fieldBizId) &&
                    fvqSettingMap.get(fieldBizId).getStatus() == 1) {
                configuredFieldIds.add(fieldBizId);
            } else {
                unconfiguredFieldIds.add(fieldBizId);
            }
        }

        // 3. 处理有配置的字段（通过接口获取）
        Map<String, GetFieldListResponse> configuredFieldsMap = processConfiguredFields(configuredFieldIds, fvqSettingMap);

        // 4. 处理无配置的字段（默认查询方式）
        Map<String, GetFieldListResponse> unconfiguredFieldsMap = processUnconfiguredFields(unconfiguredFieldIds);

        // 5. 按照请求字段ID的顺序合并结果
        for (String fieldBizId : request.getFieldBizIdList()) {
            if (configuredFieldsMap.containsKey(fieldBizId)) {
                resultList.add(configuredFieldsMap.get(fieldBizId));
            } else if (unconfiguredFieldsMap.containsKey(fieldBizId)) {
                resultList.add(unconfiguredFieldsMap.get(fieldBizId));
            } else {
                // 对于既无配置也无默认数据的字段，返回空结构
                resultList.add(createEmptyFieldResponse(fieldBizId));
            }
        }

        return Result.success(resultList);
    }

    /**
     * 根据字段ID列表查询fvq_setting配置
     * @param fieldBizIdList
     * @return
     */
    private Map<String, FvqSetting> getFvqSettingsByFieldIds(List<String> fieldBizIdList) {
        Map<String, FvqSetting> resultMap = new HashMap<>();

        try {
            List<FvqSetting> fvqSettings = iFvqSettingService.queryList(fieldBizIdList);
            if (!CollectionUtils.isEmpty(fvqSettings)) {
                resultMap = fvqSettings.stream()
                        .collect(Collectors.toMap(FvqSetting::getFieldBizId, setting -> setting));
            }
        } catch (Exception e) {
            log.error("查询fvq_setting配置失败", e);
        }

        return resultMap;
    }

    /**
     * 处理有配置的字段（通过接口获取字段值）
     * @param configuredFieldIds
     * @param fvqSettingMap
     * @return
     */
    private Map<String, GetFieldListResponse> processConfiguredFields(
            List<String> configuredFieldIds, Map<String, FvqSetting> fvqSettingMap) {
        Map<String, GetFieldListResponse> resultMap = new HashMap<>();

        if (CollectionUtils.isEmpty(configuredFieldIds)) {
            return resultMap;
        }

        // 根据字段ID列表查询字段基本信息
        List<Field> fieldList = iFieldService.queryList(
                FieldDto.builder().fieldBizIdList(configuredFieldIds).build());
        List<ApiRelFieldValueResponse> fieldBaseInfos = fieldList.stream().map(dto -> {
            ApiRelFieldValueResponse response = new ApiRelFieldValueResponse();
            BeanUtils.copyProperties(dto,response);
            return response;
        }).collect(Collectors.toList());

        if (CollectionUtils.isEmpty(fieldBaseInfos)) {
            return resultMap;
        }

        // 按字段ID分组
        Map<String, List<ApiRelFieldValueResponse>> fieldGroupMap = fieldBaseInfos.stream()
                .collect(Collectors.groupingBy(ApiRelFieldValueResponse::getFieldBizId));

        for (String fieldBizId : configuredFieldIds) {
            FvqSetting setting = fvqSettingMap.get(fieldBizId);
            if (setting == null) {
                continue;
            }

            List<ApiRelFieldValueResponse> fieldValues = fieldGroupMap.get(fieldBizId);
            if (CollectionUtils.isEmpty(fieldValues)) {
                continue;
            }

            ApiRelFieldValueResponse firstValue = fieldValues.get(0);

            // 构建字段响应对象
            GetFieldListResponse fieldResponse = new GetFieldListResponse();
            fieldResponse.setFieldBizId(firstValue.getFieldBizId());
            fieldResponse.setFieldCode(firstValue.getFieldCode());
            fieldResponse.setFieldCnName(firstValue.getFieldCnName());
            fieldResponse.setFieldEnName(firstValue.getFieldEnName());
            fieldResponse.setTextBoxType(firstValue.getTextBoxType());

            try {
                // 通过配置的接口获取字段值
                List<GetFieldValueListResponse> remoteValues = fetchFieldValuesFromRemote(setting);
                fieldResponse.setFieldValueListResponseList(remoteValues);
            } catch (Exception e) {
                log.error("通过接口获取字段值失败, fieldBizId: {}, url: {}",
                        fieldBizId, setting.getReqUrl(), e);
                // 接口调用失败时，返回空列表
                fieldResponse.setFieldValueListResponseList(new ArrayList<>());
            }

            resultMap.put(fieldBizId, fieldResponse);
        }

        return resultMap;
    }

    /**
     * 从远程接口获取字段值（支持token传递）
     * @param setting
     * @return
     * @throws Exception
     */
    private List<GetFieldValueListResponse> fetchFieldValuesFromRemote(FvqSetting setting) throws Exception {
        String url = setting.getReqUrl();
        HttpMethod method = HttpMethod.resolve(setting.getReqMethod());

        if (method == null) {
            method = HttpMethod.GET; // 默认GET请求
        }

        // 构建请求头
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        // 添加token到请求头
        String token = TokenUtils.getCurrentToken();
        if (StringUtils.isNotBlank(token)) {
            // 如果token以Bearer开头，直接使用；否则添加Bearer前缀
            if (token.startsWith("Bearer ")) {
                headers.set("Authorization", token);
            } else {
                headers.set("Authorization", "Bearer " + token);
            }
        } else {
            throw new BusinessException("token不能为空");
        }

        // 构建请求参数
        HttpEntity<String> requestEntity;
        if (StringUtils.isNotBlank(setting.getReqParam())) {
            requestEntity = new HttpEntity<>(setting.getReqParam(), headers);
        } else {
            requestEntity = new HttpEntity<>(headers);
        }

        // 发送请求
        log.debug("调用远程接口获取字段值，URL: {}, Method: {}, Headers: {}",
                url, method, headers);

        ResponseEntity<String> responseEntity;
        try {
            responseEntity = restTemplate.exchange(
                    url, method, requestEntity, String.class);
        } catch (Exception e) {
            log.error("调用远程接口失败: {}", e.getMessage(), e);
            throw new RuntimeException("调用远程接口失败: " + e.getMessage(), e);
        }

        // 处理响应状态
        if (!responseEntity.getStatusCode().is2xxSuccessful()) {
            log.error("远程接口返回错误状态码: {}, 响应体: {}",
                    responseEntity.getStatusCode(), responseEntity.getBody());
            throw new RuntimeException("远程接口返回错误状态码: " + responseEntity.getStatusCode());
        }

        // 解析响应并根据映射规则提取字段值
        return parseRemoteResponse(responseEntity.getBody(), setting.getResParamMapping());
    }

    /**
     * 解析远程响应并映射为字段值列表（增强版）
     * @param responseBody
     * @param resParamMapping
     * @return
     * @throws Exception
     */
    private List<GetFieldValueListResponse> parseRemoteResponse(
            String responseBody, String resParamMapping) throws Exception {
        List<GetFieldValueListResponse> result = new ArrayList<>();

        if (StringUtils.isBlank(resParamMapping)) {
            log.warn("res_param_mapping为空，无法解析响应");
            return result;
        }

        // 解析响应映射配置
        Map<String, String> mapping = objectMapper.readValue(
                resParamMapping, new TypeReference<Map<String, String>>() {});

        // 获取映射配置
        String listPath = mapping.getOrDefault("listPath", "data.records");
        String fieldValueBizIdKey = mapping.get("fieldValueBizId");
        String fieldValueCodeKey = mapping.get("fieldValueCode");
        String valueKey = mapping.get("value");

        if (StringUtils.isAnyBlank(fieldValueBizIdKey, valueKey)) {
            log.error("res_param_mapping配置不完整，缺少必要字段映射");
            return result;
        }

        // 解析响应数据
        Map<String, Object> responseMap = objectMapper.readValue(
                responseBody, new TypeReference<Map<String, Object>>() {});

        // 检查响应状态
        Object codeObj = responseMap.get("code");
        if (codeObj != null && !"200".equals(codeObj.toString()) && !"0".equals(codeObj.toString())) {
            log.error("远程接口返回错误码: {}, 消息: {}",
                    codeObj, responseMap.get("msg"));
            return result;
        }

        // 根据listPath获取数据列表
        List<Map<String, Object>> dataList = getDataListByPath(responseMap, listPath);

        if (!CollectionUtils.isEmpty(dataList)) {
            for (Map<String, Object> item : dataList) {
                GetFieldValueListResponse valueResponse = new GetFieldValueListResponse();

                // 根据映射规则设置字段值
                String fieldValueBizId = getValueByKey(item, fieldValueBizIdKey);
                String fieldValueCode = getValueByKey(item, fieldValueCodeKey);
                String value = getValueByKey(item, valueKey);

                // 如果fieldValueCodeKey未配置，则使用fieldValueBizId或value作为code
                if (StringUtils.isBlank(fieldValueCode) && StringUtils.isNotBlank(fieldValueBizIdKey)) {
                    fieldValueCode = fieldValueBizId;
                }

                valueResponse.setFieldValueBizId(fieldValueBizId);
                valueResponse.setFieldValueCode(fieldValueCode);
                valueResponse.setValue(value);

                result.add(valueResponse);
            }
        }

        return result;
    }

    /**
     * 根据路径获取数据列表
     */
    private List<Map<String, Object>> getDataListByPath(Map<String, Object> dataMap, String path) {
        if (StringUtils.isBlank(path)) {
            return Collections.emptyList();
        }

        String[] keys = path.split("\\.");
        Object current = dataMap;

        try {
            for (String key : keys) {
                if (current instanceof Map) {
                    current = ((Map<?, ?>) current).get(key);
                } else {
                    return Collections.emptyList();
                }
            }

            if (current instanceof List) {
                @SuppressWarnings("unchecked")
                List<Map<String, Object>> result = (List<Map<String, Object>>) current;
                return result;
            }
        } catch (Exception e) {
            log.error("解析listPath失败: {}", path, e);
        }

        return Collections.emptyList();
    }

    /**
     * 根据key获取值（支持嵌套路径）
     */
    private String getValueByKey(Map<String, Object> item, String key) {
        if (StringUtils.isBlank(key)) {
            return null;
        }

        String[] keys = key.split("\\.");
        Object current = item;

        try {
            for (String k : keys) {
                if (current instanceof Map) {
                    current = ((Map<?, ?>) current).get(k);
                } else {
                    return null;
                }
            }

            return current != null ? current.toString() : null;
        } catch (Exception e) {
            log.error("获取字段值失败: {}", key, e);
            return null;
        }
    }

    /**
     * 处理无配置的字段（默认查询方式）
     * @param unconfiguredFieldIds
     * @return
     */
    private Map<String, GetFieldListResponse> processUnconfiguredFields(List<String> unconfiguredFieldIds) {
        Map<String, GetFieldListResponse> resultMap = new HashMap<>();

        if (CollectionUtils.isEmpty(unconfiguredFieldIds)) {
            return resultMap;
        }

        // 根据字段表唯一业务ID列表查询字段列表信息（默认方式）
        List<ApiRelFieldValueResponse> responses = iRelFieldValueService.list(
                RelFieldValueDto.builder().fieldBizIdList(unconfiguredFieldIds).build());

        if (CollectionUtils.isEmpty(responses)) {
            return resultMap;
        }

        // 按字段ID分组
        Map<String, List<ApiRelFieldValueResponse>> fieldGroupMap = responses.stream()
                .collect(Collectors.groupingBy(ApiRelFieldValueResponse::getFieldBizId));

        for (String fieldBizId : unconfiguredFieldIds) {
            List<ApiRelFieldValueResponse> fieldValues = fieldGroupMap.get(fieldBizId);

            if (CollectionUtils.isEmpty(fieldValues)) {
                continue;
            }

            ApiRelFieldValueResponse firstValue = fieldValues.get(0);

            GetFieldListResponse fieldResponse = new GetFieldListResponse();
            // 设置字段基本信息
            fieldResponse.setFieldBizId(firstValue.getFieldBizId());
            fieldResponse.setFieldCode(firstValue.getFieldCode());
            fieldResponse.setFieldCnName(firstValue.getFieldCnName());
            fieldResponse.setFieldEnName(firstValue.getFieldEnName());
            fieldResponse.setTextBoxType(firstValue.getTextBoxType());

            // 构建字段值列表
            List<GetFieldValueListResponse> valueList = fieldValues.stream()
                    .map(response -> {
                        GetFieldValueListResponse valueResponse = new GetFieldValueListResponse();
                        valueResponse.setFieldValueBizId(response.getFieldValueBizId());
                        valueResponse.setFieldValueCode(response.getFieldValueCode());
                        valueResponse.setValue(response.getValue());
                        return valueResponse;
                    })
                    .collect(Collectors.toList());

            fieldResponse.setFieldValueListResponseList(valueList);
            resultMap.put(fieldBizId, fieldResponse);
        }

        return resultMap;
    }

    /**
     * 创建空的字段响应（用于没有数据的字段）
     * @param fieldBizId
     * @return
     */
    private GetFieldListResponse createEmptyFieldResponse(String fieldBizId) {
        GetFieldListResponse fieldResponse = new GetFieldListResponse();
        fieldResponse.setFieldBizId(fieldBizId);
        fieldResponse.setFieldValueListResponseList(new ArrayList<>());
        return fieldResponse;
    }
}