package com.yd.base.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.base.api.service.ApiFieldValueService;
import com.yd.base.api.service.ApiRelFieldValueService;
import com.yd.base.feign.request.fieldvalue.ApiFieldValueAddRequest;
import com.yd.base.feign.request.fieldvalue.ApiFieldValueEditRequest;
import com.yd.base.feign.request.fieldvalue.ApiFieldValueListRequest;
import com.yd.base.feign.request.fieldvalue.ApiFieldValuePageRequest;
import com.yd.base.feign.response.fieldvalue.ApiFieldValueDetailResponse;
import com.yd.base.feign.response.fieldvalue.ApiFieldValuePageResponse;
import com.yd.base.service.dto.FieldValueDto;
import com.yd.base.service.model.FieldValue;
import com.yd.base.service.service.IFieldValueService;
import com.yd.common.constant.CodeGeneratorConstants;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.CodeGenerator;
import com.yd.common.utils.RandomStringGenerator;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiFieldValueServiceImpl implements ApiFieldValueService {

    @Autowired
    private IFieldValueService iFieldValueService;

    @Autowired
    private ApiRelFieldValueService apiRelFieldValueService;

    /**
     * 分页列表查询-字段值
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiFieldValuePageResponse>> page(ApiFieldValuePageRequest request) {
        Page<ApiFieldValuePageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiFieldValuePageResponse> iPage = iFieldValueService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 列表查询-字段值
     * @param request
     * @return
     */
    @Override
    public Result<List<ApiFieldValueDetailResponse>> list(ApiFieldValueListRequest request) {
        List<ApiFieldValueDetailResponse> responses = new ArrayList<>();
        List<FieldValue> fieldValueList = iFieldValueService.queryList(FieldValueDto.builder().fieldValueBizIdList(request.getFieldValueBizIdList()).build());
        if (!CollectionUtils.isEmpty(fieldValueList)) {
            responses = fieldValueList.stream().map(dto -> {
                ApiFieldValueDetailResponse response = new ApiFieldValueDetailResponse();
                BeanUtils.copyProperties(dto,response);
                return response;
            }).collect(Collectors.toList());
        }
        return Result.success(responses);
    }

    /**
     * 添加-字段值
     * @param request
     * @return
     */
    @Override
    public Result add(ApiFieldValueAddRequest request) {
        //校验字段值的唯一性
        List<FieldValue> fieldValueList =  iFieldValueService.queryList(FieldValueDto.builder()
                .value(request.getValue())
                .build());
        if (!CollectionUtils.isEmpty(fieldValueList)) {
            throw new BusinessException("字段值已存在");
        }
        FieldValue fieldValue = new FieldValue();
        BeanUtils.copyProperties(request,fieldValue);
        //字段值表唯一业务ID
        fieldValue.setFieldValueBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_FIELD_VALUE.getCode()));
        //字段值编码
        fieldValue.setFieldValueCode(CodeGenerator.generate10Code(CodeGeneratorConstants.PREFIX_FIELD_VALUE_CODE));
        iFieldValueService.saveOrUpdate(fieldValue);

        //保存绑定的字段表唯一业务ID列表
        apiRelFieldValueService.save(fieldValue.getFieldValueBizId(),request.getFieldBizIdList());
        return Result.success();
    }

    /**
     * 编辑-字段值
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiFieldValueEditRequest request) {
        //校验字段值对象是否存在
        Result<FieldValue> result = checkFieldValueIsExist(request.getFieldValueBizId());
        FieldValue fieldValue = result.getData();
        //校验字段值的唯一性
        List<FieldValue> fieldValueList =  iFieldValueService.queryList(FieldValueDto.builder()
                .value(request.getValue())
                .fieldValueBizId(request.getFieldValueBizId())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(fieldValueList)) {
            throw new BusinessException("字段值已存在");
        }
        BeanUtils.copyProperties(request,fieldValue);
        iFieldValueService.saveOrUpdate(fieldValue);

        //保存绑定的字段表唯一业务ID列表
        apiRelFieldValueService.save(fieldValue.getFieldValueBizId(),request.getFieldBizIdList());
        return Result.success();
    }

    /**
     * 详情-字段值
     * @param fieldValueBizId 字段值表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiFieldValueDetailResponse> detail(String fieldValueBizId) {
        //校验字段值对象是否存在
        Result<FieldValue> result = checkFieldValueIsExist(fieldValueBizId);
        FieldValue fieldValue = result.getData();
        ApiFieldValueDetailResponse response = new ApiFieldValueDetailResponse();
        BeanUtils.copyProperties(fieldValue,response);
        return Result.success(response);
    }

    /**
     * 删除-字段值
     * @param fieldValueBizId 字段值表唯一业务ID
     * @return
     */
    @Override
    public Result del(String fieldValueBizId) {
        return null;
    }

    /**
     * 校验字段值对象是否存在
     * @param fieldValueBizId
     * @return
     */
    public Result<FieldValue> checkFieldValueIsExist(String fieldValueBizId) {
        FieldValue fieldValue = iFieldValueService.queryOne(fieldValueBizId);
        if (Objects.isNull(fieldValue)) {
            //数据不存在
            throw new BusinessException("字段值数据不存在");
        }
        return Result.success(fieldValue);
    }
}
