package com.yd.gateway.config;

import com.yd.gateway.exception.JwtAuthException;
import com.yd.gateway.utils.JwtTokenUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.gateway.filter.GlobalFilter;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpStatus;
import org.springframework.util.AntPathMatcher;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

/**
 * 网关JWT认证过滤器：只做轻量校验（验证token有效性），不查询业务数据
 */
@Configuration
@Slf4j
public class GatewayJwtAuthFilterConfig {

    @Autowired
    private JwtTokenUtil jwtTokenUtil;

    private AntPathMatcher matcher = new AntPathMatcher();

    private static final String[] KNIFE4J_WHITELIST = {
            "/doc.html",
            "/webjars/**",
            "/v3/api-docs/*",
            "/swagger-resources/*",
            "/swagger-ui/*",
            "/auth/v2/api-docs/*",
            "/favicon.ico",
            "/user/api/v3/api-docs/**",
            "/insurance/base/api/v3/api-docs/**",
            "/csf/api/v3/api-docs/**"
    };

    /**
     * 注册全局过滤器，优先级高于路由过滤器（确保先校验再路由）
     */
    @Bean
    @Order(Ordered.HIGHEST_PRECEDENCE)  // 最高优先级，在路由前执行
    public GlobalFilter jwtAuthFilter() {
        return (exchange, chain) -> {
            // 1. 获取请求路径，跳过白名单（如登录、Swagger等）
            String path = exchange.getRequest().getPath().toString();
            if (isWhitelist(path)) {
                // 白名单路径直接放行
                return chain.filter(exchange);
            }

            // 2. 从请求头获取token
            String token = resolveToken(exchange);
            if (token == null) {
                // 无token，返回401未授权
                return setUnauthorizedResponse(exchange, "未提供令牌（Authorization头缺失或格式错误）");
            }

            // 3. 验证token有效性（只做签名、过期等基础校验，不查用户）
            try {
                jwtTokenUtil.validateToken(token);
                // token有效，继续路由到下游服务
                return chain.filter(exchange);
            } catch (JwtAuthException e) {
                // token无效（如过期、签名错误），返回401
                return setUnauthorizedResponse(exchange, "令牌无效：" + e.getMessage());
            }
        };
    }

    /**
     * 解析请求头中的token（Authorization: Bearer <token>）
     */
    private String resolveToken(ServerWebExchange exchange) {
        String authHeader = exchange.getRequest().getHeaders().getFirst("Authorization");
        if (authHeader != null && authHeader.startsWith("Bearer ")) {
            return authHeader.substring(7).trim();  // 去除"Bearer "前缀
        }
        return null;
    }

    private boolean isKnife4jWhitelist(String requestPath) {
        for (String pattern : KNIFE4J_WHITELIST) {
            if (matcher.match(pattern, requestPath)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 判断路径是否在白名单中（无需认证）
     */
    private boolean isWhitelist(String path) {
        // 白名单路径：与SecurityWebFilterChain中的配置保持一致
        return path.startsWith("/auth/")
                || path.startsWith("/swagger-ui/")
                || path.startsWith("/v3/api-docs/")
                || isKnife4jWhitelist(path);
    }

    /**
     * 设置401响应
     */
    private Mono<Void> setUnauthorizedResponse(ServerWebExchange exchange, String message) {
        exchange.getResponse().setStatusCode(HttpStatus.UNAUTHORIZED);
        exchange.getResponse().getHeaders().add("Content-Type", "application/json;charset=UTF-8");
        // 构建错误响应体
        String json = "{\"status\":401,\"error\":\"Unauthorized\",\"message\":\"" + message + "\"}";
        return exchange.getResponse()
                .writeWith(Mono.just(exchange.getResponse().bufferFactory().wrap(json.getBytes())));
    }
}
