package com.yd.gateway.config;

import com.yd.auth.core.security.JwtTokenProvider;
import com.yd.auth.core.security.JwtTokenProvider.JwtAuthenticationException;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.config.annotation.web.reactive.EnableWebFluxSecurity;
import org.springframework.security.config.web.server.ServerHttpSecurity;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.security.web.server.SecurityWebFilterChain;
import reactor.core.publisher.Mono;

@Configuration
@EnableWebFluxSecurity
public class GatewaySecurityConfig {

    private final JwtTokenProvider jwtTokenProvider;

    public GatewaySecurityConfig(JwtTokenProvider jwtTokenProvider) {
        this.jwtTokenProvider = jwtTokenProvider;
    }

    @Bean
    public SecurityWebFilterChain securityWebFilterChain(ServerHttpSecurity http) {
        http
                .csrf().disable()
                .authorizeExchange(exchanges -> exchanges
                        .pathMatchers(
                                "/auth/**",
                                "/swagger-ui/**",
                                "/v3/api-docs/**"
                        ).permitAll() // 放行认证和Swagger路径
                        .anyExchange().authenticated() // 其他请求需要认证
                )
                .oauth2ResourceServer(oauth2 -> oauth2
                        .jwt(jwt -> jwt
                                .jwtDecoder(token -> {
                                    try {
                                        // 验证令牌有效性
                                        jwtTokenProvider.validateToken(token);
                                        // 将字符串转换为 Jwt 对象
                                        Jwt jwtObject = Jwt.withTokenValue(token).build();
                                        return Mono.just(jwtObject);
                                    } catch (JwtAuthenticationException e) {
                                        return Mono.error(e);
                                    }
                                })
                        )
                );
        return http.build();
    }
}
