package com.yd.auth.core.security;

import com.yd.auth.core.dto.AuthUserDto;
import com.yd.common.result.Result;
import com.yd.user.feign.client.sysuser.ApiSysUserFeignClient;
import com.yd.user.feign.request.sysuser.ApiSysUserDetailBodyRequest;
import com.yd.user.feign.response.sysuser.ApiSysUserDetailResponse;
import com.yd.user.service.model.SysUser;
import com.yd.user.service.service.ISysUserService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Objects;

@Component
public class JwtAuthenticationFilter extends OncePerRequestFilter {

//    @Autowired
//    @Qualifier("sysUserServiceImpl")
//    private ISysUserService iSysUserService;

    @Autowired
    private ApiSysUserFeignClient apiSysUserFeignClient;

    private final JwtTokenProvider tokenProvider;

    public JwtAuthenticationFilter(JwtTokenProvider tokenProvider) {
        this.tokenProvider = tokenProvider;
    }

    @Override
    protected void doFilterInternal(HttpServletRequest request,
                                    HttpServletResponse response,
                                    FilterChain filterChain) throws ServletException, IOException {

        // 1. 从HTTP请求中解析JWT令牌（通常从Authorization头获取）
        String token = resolveToken(request);

        // 2. 检查令牌是否存在且有效
        if (token != null && tokenProvider.validateToken(token)) {
            // 3. 从JWT令牌中提取用户唯一标识(业务ID)
            String userBizId = tokenProvider.getUserBizIdFromToken(token);

            // 4. 根据用户唯一标识查询用户详细信息（包括权限信息）
            UserDetails userDetails = queryUserDetails(userBizId);

            // 5. 创建Spring Security认证对象
            //    - userDetails: 包含用户身份和权限信息
            //    - null: 凭证（密码），JWT认证中不需要
            //    - userDetails.getAuthorities(): 用户的权限集合
            UsernamePasswordAuthenticationToken authentication =
                    new UsernamePasswordAuthenticationToken(userDetails, null, userDetails.getAuthorities());

            // 6. 将认证对象设置到SecurityContext中，表示用户已认证，这里设置后续接口能直接拿到登录用户信息
            SecurityContextHolder.getContext().setAuthentication(authentication);
        }

        // 7. 继续执行后续过滤器链（无论是否认证都要继续处理请求）
        filterChain.doFilter(request, response);
    }

    /**
     * 查询用户信息返回spring security认证对象UserDetails
     * @param userBizId
     * @return spring security认证对象UserDetails
     */
    public UserDetails queryUserDetails(String userBizId) {
        AuthUserDto authUserDto = null;

        // 通过Feign调用yd-user-feign服务获取用户信息
        ApiSysUserDetailBodyRequest request = new ApiSysUserDetailBodyRequest();
        request.setUserBizId(userBizId);
        Result<ApiSysUserDetailResponse> result = apiSysUserFeignClient.detail(request);
        if (Objects.isNull(result.getData())) {
            throw new UsernameNotFoundException("用户不存在:");
        }
        ApiSysUserDetailResponse response = result.getData();
        authUserDto = new AuthUserDto();
        BeanUtils.copyProperties(response,authUserDto);
        authUserDto.setUsername(response.getUserName());
        // 查询用户角色 TODO
//        authUserDto.setRoles(roles);

        return authUserDto;

    }

//    /**
//     * 查询用户信息返回spring security认证对象UserDetails
//     * @param userBizId
//     * @return spring security认证对象UserDetails
//     */
//    public UserDetails queryUserDetails(String userBizId) {
//        AuthUserDto authUserDto = null;
//
//        SysUser sysUser = iSysUserService.queryOne(userBizId);
//        if (Objects.isNull(sysUser)) {
//            throw new UsernameNotFoundException("用户不存在:");
//        }
//        authUserDto = new AuthUserDto();
//        BeanUtils.copyProperties(sysUser,authUserDto);
//        authUserDto.setUsername(sysUser.getUserName());
//        // 查询用户角色 TODO
////        authUserDto.setRoles(roles);
//
//        return authUserDto;
//
//    }

    private String resolveToken(HttpServletRequest request) {
        String bearerToken = request.getHeader("Authorization");
        if (StringUtils.hasText(bearerToken) && bearerToken.startsWith("Bearer ")) {
            return bearerToken.substring(7);
        }
        return bearerToken;
    }
}
