package com.yd.framework.handler;

import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import lombok.extern.slf4j.Slf4j;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;

import javax.validation.ConstraintViolationException;
import java.util.stream.Collectors;

//import com.yd.common.model.R;
//import org.springframework.http.HttpStatus;
//import org.springframework.security.authentication.BadCredentialsException;
//import org.springframework.web.bind.annotation.ExceptionHandler;
//import org.springframework.web.bind.annotation.ResponseStatus;
//import org.springframework.web.bind.annotation.RestControllerAdvice;

/**
 * 全局异常处理器
 */
@Slf4j
@RestControllerAdvice
public class GlobalExceptionHandler {

    /**
     * 处理业务异常
     */
    @ExceptionHandler(BusinessException.class)
    public Result<?> handleBusinessException(BusinessException e) {
        log.error("业务异常: {}", e.getMessage());
        return Result.fail(e.getCode(), e.getMessage());
    }

    /**
     * @Validated校验器触发的异常  @RequestBody对象校验
     * 处理参数校验异常
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public Result<?> handleMethodArgumentNotValidException(MethodArgumentNotValidException e) {
        BindingResult bindingResult = e.getBindingResult();
        FieldError fieldError = bindingResult.getFieldError();
        String message = fieldError != null ? fieldError.getDefaultMessage() : "参数校验失败";
        log.error("RequestBody参数校验异常: {}", message);
        return Result.fail(ResultCode.PARAM_CHECK_ERROR.getCode(), message);
    }

    /**
     * @Validated校验器触发的异常  RequestParam 参数校验
     * @param e
     * @return
     */
    @ExceptionHandler(ConstraintViolationException.class)
    public Result<?> constraintViolationExceptionHandler(ConstraintViolationException e){
        String message = e.getConstraintViolations().stream().collect(Collectors.toList()).get(0).getMessage();

        log.error("RequestParam参数校验异常：{}",e);
        return Result.fail(ResultCode.PARAM_CHECK_ERROR.getCode(),message);
    }

    // 专门捕获 BadCredentialsException，返回“密码错误”的自定义响应
    @ExceptionHandler(BadCredentialsException.class)
    public Result<?> handleBadCredentialsException(BadCredentialsException e) {
        log.error("认证失败：{}", e.getMessage());
        // 返回自定义的错误码和“密码错误”提示（而非模糊的“用户名或密码错误”）
        return Result.fail("认证失败：密码错误，请重新输入");
    }

    /**
     * 处理其他异常
     */
    @ExceptionHandler(Exception.class)
    public Result<?> handleException(Exception e) {
        log.error("系统异常: ", e);
        return Result.fail("系统异常，请联系管理员");
    }




//    @ExceptionHandler(BadCredentialsException.class)
//    @ResponseStatus(HttpStatus.UNAUTHORIZED)
//    public R<?> handleBadCredentialsException(BadCredentialsException e) {
//        return R.fail(401, "用户名或密码错误");
//    }
//
//    @ExceptionHandler(JwtTokenProvider.JwtAuthenticationException.class)
//    @ResponseStatus(HttpStatus.UNAUTHORIZED)
//    public R<?> handleJwtAuthenticationException(JwtTokenProvider.JwtAuthenticationException e) {
//        return R.fail(401, e.getMessage());
//    }
}
