package com.yd.auth.core.service.impl;

import com.yd.auth.core.config.LoginConfig;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.request.LoginRequest;
import com.yd.auth.core.response.LoginResponse;
import com.yd.auth.core.security.JwtTokenProvider;
import com.yd.auth.core.service.AuthService;
import com.yd.common.constant.RedisConstants;
import com.yd.common.utils.RedisUtil;
import com.yd.common.utils.SM3Util;
import com.yd.user.service.model.SysUser;
import com.yd.user.service.service.ISysUserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Service;

import java.util.Objects;
import java.util.concurrent.TimeUnit;

@Service
public class AuthServiceImpl implements AuthService {

    @Autowired
    @Qualifier("sysUserServiceImpl")
    private ISysUserService iSysUserService;

    @Autowired
    private RedisUtil redisUtil;

    @Autowired
    private LoginConfig loginConfig;

    private final AuthenticationManager authenticationManager;
    private final JwtTokenProvider jwtTokenProvider;

    @Autowired
    public AuthServiceImpl(AuthenticationManager authenticationManager, 
                          JwtTokenProvider jwtTokenProvider) {
        this.authenticationManager = authenticationManager;
        this.jwtTokenProvider = jwtTokenProvider;
    }

    /**
     * 用户登录认证方法（基于Spring Security的认证流程）
     *
     * @param loginRequest 登录请求对象（包含用户名和密码）
     * @return 登录响应对象（包含JWT令牌和过期时间）
     */
    @Override
    public LoginResponse login(LoginRequest loginRequest) {

        // 通过Feign调用yd-user-service服务获取用户信息
        SysUser sysUser = iSysUserService.queryOneByName(loginRequest.getUsername());
        if (Objects.isNull(sysUser)) {
            throw new UsernameNotFoundException("用户不存在");
        }

        //密码：SM3国密算法
        //生成盐值
        String salt = sysUser.getPasswordSalt();
        //生成加密密码：带盐的SM3哈希
        String hashedPassword = SM3Util.hashWithSalt(loginRequest.getPassword(), salt);

        // 1. 创建未认证的Authentication对象
        //    - 第一个参数：用户名（作为身份主体）
        //    - 第二个参数：原始密码（作为凭证）
        Authentication authentication = new UsernamePasswordAuthenticationToken(
                loginRequest.getUsername(),
                hashedPassword
        );

        // 2. 执行Spring Security认证流程（核心步骤）
        //    - 触发AuthUserDetailsService.loadUserByUsername()方法
        //    - 自动进行密码比对（使用配置的PasswordEncoder）
        //    - 成功返回已认证的Authentication对象，失败抛出AuthenticationException
        Authentication authenticated = authenticationManager.authenticate(authentication);

        // 3. 将认证信息设置到SecurityContext
        //    - 使当前请求线程具有认证上下文
        //    - 后续操作可通过SecurityContextHolder获取当前用户
        SecurityContextHolder.getContext().setAuthentication(authenticated);

        // 4. 生成JWT令牌
        //    - 基于认证对象中的用户信息生成令牌
        //    - 令牌中包含用户标识和权限信息
        String token = jwtTokenProvider.generateToken(authenticated);

        // 5. 构建登录响应对象
        LoginResponse response = new LoginResponse();
        response.setToken(token);                   // 设置生成的JWT令牌
        response.setExpiresIn(jwtTokenProvider.getJwtExpiration());  // 设置令牌过期时间

        //获取用户登录信息存入redis缓存中,2天
        AuthUserDto currentUser = (AuthUserDto) authenticated.getPrincipal();
        redisUtil.setCacheObject(RedisConstants.LOGIN_USER + sysUser.getUserBizId(),currentUser,loginConfig.getRedisTimeout(), TimeUnit.DAYS);
        return response;
    }

}
