package com.yd.common.utils;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.UUID;

/**
 * UUID生成工具类
 * 提供多种UUID生成方式
 */
public class UUIDUtil {
    
    private UUIDUtil() {
        // 工具类，防止实例化
    }
    
    /**
     * 生成随机UUID (版本4)
     * @return 随机UUID字符串
     */
    public static String randomUUID() {
        return UUID.randomUUID().toString();
    }
    
    /**
     * 生成不带连字符的随机UUID
     * @return 不带连字符的UUID字符串
     */
    public static String randomUUIDWithoutHyphen() {
        return UUID.randomUUID().toString().replace("-", "");
    }
    
    /**
     * 生成大写UUID
     * @return 大写UUID字符串
     */
    public static String randomUUIDUpperCase() {
        return UUID.randomUUID().toString().toUpperCase();
    }
    
    /**
     * 生成基于名称的UUID (版本3 - MD5)
     * @param name 名称
     * @return 基于名称的UUID
     */
    public static String nameBasedUUID(String name) {
        return UUID.nameUUIDFromBytes(name.getBytes()).toString();
    }
    
    /**
     * 生成基于名称空间的UUID (版本5 - SHA-1)
     * @param namespace 命名空间
     * @param name 名称
     * @return 基于名称空间的UUID
     */
    public static String namespaceBasedUUID(String namespace, String name) {
        try {
            // 将命名空间转换为UUID字节
            byte[] namespaceBytes = UUID.fromString(namespace).toString().getBytes();
            
            MessageDigest md = MessageDigest.getInstance("SHA-1");
            md.update(namespaceBytes);
            byte[] hash = md.digest(name.getBytes());
            
            // 设置版本号为5
            hash[6] &= 0x0F;
            hash[6] |= 0x50;
            // 设置变体
            hash[8] &= 0x3F;
            hash[8] |= 0x80;
            
            return bytesToUUID(hash);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("SHA-1 algorithm not available", e);
        }
    }
    
    /**
     * 生成时间序UUID (版本1)
     * 模拟版本1UUID，包含时间戳和随机节点信息
     * @return 时间序UUID
     */
    public static String timeBasedUUID() {
        try {
            SecureRandom secureRandom = SecureRandom.getInstanceStrong();
            
            long time = System.currentTimeMillis();
            long timeLow = (time & 0xFFFFFFFFL) << 32;
            long timeMid = ((time >> 32) & 0xFFFFL) << 16;
            long timeHi = ((time >> 48) & 0x0FFFL) | 0x1000L; // 设置版本号为1
            
            long mostSigBits = timeLow | timeMid | timeHi;
            
            long leastSigBits = secureRandom.nextLong();
            leastSigBits = (leastSigBits & 0x3FFFFFFFFFFFFFFFL) | 0x8000000000000000L; // 设置变体
            
            UUID uuid = new UUID(mostSigBits, leastSigBits);
            return uuid.toString();
        } catch (NoSuchAlgorithmException e) {
            return UUID.randomUUID().toString();
        }
    }
    
    /**
     * 生成短UUID (8位)
     * @return 8位短UUID
     */
    public static String shortUUID() {
        UUID uuid = UUID.randomUUID();
        long mostSigBits = uuid.getMostSignificantBits();
        long leastSigBits = uuid.getLeastSignificantBits();
        
        // 取部分位进行组合
        long combined = mostSigBits ^ leastSigBits;
        return Long.toHexString(combined).substring(0, 8);
    }
    
    /**
     * 验证UUID格式是否有效
     * @param uuid 待验证的UUID字符串
     * @return 是否有效
     */
    public static boolean isValidUUID(String uuid) {
        if (uuid == null) {
            return false;
        }
        
        try {
            UUID.fromString(uuid);
            return true;
        } catch (IllegalArgumentException e) {
            // 检查不带连字符的格式
            if (uuid.length() == 32) {
                try {
                    String formatted = uuid.substring(0, 8) + "-" +
                                     uuid.substring(8, 12) + "-" +
                                     uuid.substring(12, 16) + "-" +
                                     uuid.substring(16, 20) + "-" +
                                     uuid.substring(20, 32);
                    UUID.fromString(formatted);
                    return true;
                } catch (IllegalArgumentException ex) {
                    return false;
                }
            }
            return false;
        }
    }
    
    /**
     * 将字节数组转换为UUID
     * @param bytes 字节数组(长度至少为16)
     * @return UUID字符串
     */
    private static String bytesToUUID(byte[] bytes) {
        if (bytes.length < 16) {
            throw new IllegalArgumentException("Byte array must be at least 16 bytes long");
        }
        
        long mostSigBits = 0;
        long leastSigBits = 0;
        
        for (int i = 0; i < 8; i++) {
            mostSigBits = (mostSigBits << 8) | (bytes[i] & 0xff);
        }
        for (int i = 8; i < 16; i++) {
            leastSigBits = (leastSigBits << 8) | (bytes[i] & 0xff);
        }
        
        return new UUID(mostSigBits, leastSigBits).toString();
    }
    
    /**
     * 获取UUID的版本号
     * @param uuid UUID字符串
     * @return 版本号，如果无效返回-1
     */
    public static int getVersion(String uuid) {
        try {
            UUID u = UUID.fromString(uuid);
            return u.version();
        } catch (IllegalArgumentException e) {
            return -1;
        }
    }
}