package com.yd.auth.core.security;

import com.yd.auth.core.dto.AuthUserDto;
import io.jsonwebtoken.*;
import io.jsonwebtoken.io.Decoders;
import io.jsonwebtoken.security.Keys;
import io.jsonwebtoken.security.SecurityException;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.stereotype.Component;
import javax.crypto.SecretKey;
import java.util.Collection;
import java.util.Date;
import java.util.Base64;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * JWT令牌提供者
 * 修复了密钥长度不足的问题，并优化了异常处理
 */
@Component
public class JwtTokenProvider {

    // 使用512位长度的密钥（通过配置文件设置）
    @Value("${jwt.secret}")
    private String jwtSecret;

    @Value("${jwt.expiration}") // 默认24小时(单位毫秒)
    private long jwtExpiration;

    /**
     * 生成JWT令牌
     */
    public String generateToken(Authentication authentication) {
        if (authentication == null || authentication.getPrincipal() == null) {
            throw new JwtAuthenticationException("无法从空认证信息生成令牌");
        }

        AuthUserDto authUserDto = (AuthUserDto) authentication.getPrincipal();

        try {
            return Jwts.builder()
                    .setSubject(authUserDto.getUserBizId())
                    .claim("roles", getRoles(authUserDto.getAuthorities()))
                    .setIssuedAt(new Date())
                    .setExpiration(new Date(System.currentTimeMillis() + jwtExpiration))
                    .signWith(getSigningKey(), SignatureAlgorithm.HS512)
                    .compact();
        } catch (Exception e) {
            throw new JwtAuthenticationException("生成JWT令牌失败: " + e.getMessage());
        }
    }

    /**
     * 提取角色信息
     */
    private Collection<String> getRoles(Collection<? extends GrantedAuthority> authorities) {
        return authorities.stream()
                .map(GrantedAuthority::getAuthority)
                .collect(Collectors.toList());
    }

    /**
     * 从令牌中获取用户名
     */
    public String getUsernameFromToken(String token) {
        try {
            return Jwts.parserBuilder()
                    .setSigningKey(getSigningKey())
                    .build()
                    .parseClaimsJws(token)
                    .getBody()
                    .getSubject();
        } catch (Exception e) {
            throw new JwtAuthenticationException("从令牌获取用户名失败: " + e.getMessage());
        }
    }

    /**
     * 从令牌中获取用户业务id
     */
    public String getUserBizIdFromToken(String token) {
        try {
            return Jwts.parserBuilder()
                    .setSigningKey(getSigningKey())
                    .build()
                    .parseClaimsJws(token)
                    .getBody()
                    .getSubject();
        } catch (Exception e) {
            throw new JwtAuthenticationException("从令牌获取用户名失败: " + e.getMessage());
        }
    }

    /**
     * 从令牌中获取认证信息
     */
    public Authentication getAuthentication(String token) {
        try {
            if (validateToken(token)) {
                String username = getUsernameFromToken(token);
                return new UsernamePasswordAuthenticationToken(
                        username,
                        null,
                        getAuthoritiesFromToken(token)
                );
            }
            return null;
        } catch (Exception e) {
            throw new JwtAuthenticationException("获取认证信息失败: " + e.getMessage());
        }
    }

    /**
     * 从令牌中获取权限信息
     */
    private Collection<? extends GrantedAuthority> getAuthoritiesFromToken(String token) {
        try {
            Claims claims = Jwts.parserBuilder()
                    .setSigningKey(getSigningKey())
                    .build()
                    .parseClaimsJws(token)
                    .getBody();

            @SuppressWarnings("unchecked")
            Collection<String> roles = (Collection<String>) claims.get("roles");

            return roles.stream()
                    .map(role -> (GrantedAuthority) () -> role)
                    .collect(Collectors.toList());
        } catch (Exception e) {
            throw new JwtAuthenticationException("从令牌获取权限信息失败: " + e.getMessage());
        }
    }

    /**
     * 验证令牌有效性
     */
    public boolean validateToken(String token) {
        try {
            Jwts.parserBuilder().setSigningKey(getSigningKey()).build().parseClaimsJws(token);
            return true;
        } catch (SecurityException | MalformedJwtException e) {
            throw new JwtAuthenticationException("无效的JWT签名");
        } catch (ExpiredJwtException e) {
            throw new JwtAuthenticationException("JWT令牌已过期");
        } catch (UnsupportedJwtException e) {
            throw new JwtAuthenticationException("不支持的JWT令牌");
        } catch (IllegalArgumentException e) {
            throw new JwtAuthenticationException("JWT令牌无效");
        } catch (Exception e) {
            throw new JwtAuthenticationException("无效的JWT令牌: " + e.getMessage());
        }
    }

    /**
     * 获取签名密钥
     */
    private SecretKey getSigningKey() {
        try {
            if (jwtSecret == null || jwtSecret.trim().isEmpty()) {
                throw new JwtAuthenticationException("JWT密钥未配置");
            }

            byte[] keyBytes = Decoders.BASE64.decode(jwtSecret);
            return Keys.hmacShaKeyFor(keyBytes);
        } catch (Exception e) {
            throw new JwtAuthenticationException("获取签名密钥失败: " + e.getMessage());
        }
    }

    /**
     * 生成符合HS512要求的密钥（用于生成配置文件中的密钥）
     */
    public static String generateSecureKey() {
        try {
            SecretKey key = Keys.secretKeyFor(SignatureAlgorithm.HS512);
            return Base64.getEncoder().encodeToString(key.getEncoded());
        } catch (Exception e) {
            throw new RuntimeException("生成安全密钥失败", e);
        }
    }

    public long getJwtExpiration() {
        return jwtExpiration;
    }

    /**
     * JWT认证异常
     */
    public static class JwtAuthenticationException extends RuntimeException {
        public JwtAuthenticationException(String message) {
            super(message);
        }

        public JwtAuthenticationException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    // 解析 JWT 声明
    public Claims parseClaims(String token) {
        return Jwts.parserBuilder()
                .setSigningKey(getSigningKey())
                .build()
                .parseClaimsJws(token)
                .getBody();
    }

    // 获取 JWT 头部信息
    public Map<String, Object> getHeaders(String token) {
        return Jwts.parserBuilder()
                .setSigningKey(getSigningKey())
                .build()
                .parseClaimsJws(token)
                .getHeader();
    }

    public static void main(String[] args) {
        // 生成一个符合HS512要求的密钥
        System.out.println("请将以下生成的JWT密钥复制到配置文件中（application.yml或application.properties）:");
        System.out.println("jwt.secret: " + generateSecureKey());
        System.out.println("jwt.expiration: 86400000 # 24小时");
    }
}
