package com.yd.common.exception;

import com.yd.common.enums.ErrorCode;
import com.yd.common.utils.ResultUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import javax.validation.ConstraintViolationException;
import java.util.stream.Collectors;

/**
 * 全局异常处理器
 * Slf4j是用来记录日志信息的，lombok中自带的。
 *
 * @author zxm
 */
@RestControllerAdvice
@Slf4j
public class GlobalExceptionHandler {

    /**
     * 处理业务返回的异常
     * @param e
     * @return
     */
    @ExceptionHandler(BusinessException.class)
    public BaseResponse businessExceptionHandler(BusinessException e){
        log.error("BusinessException：{}",e);
        return ResultUtils.error(e.getCode(),e.getMessage(),e.getDescription());
//        throw new BusinessException(e.getMessage(),e.getCode(),e.getDescription());
    }

    /**
     * 处理系统异常
     * @param e
     * @return
     */
    @ExceptionHandler(Exception.class)
    public BaseResponse exceptionHandler(Exception e){
        log.error("Exception：{}",e);
        //系统异常
        return ResultUtils.error(ErrorCode.SYSTEM_ERROR);
    }

    /**
     * @Validated校验器触发的异常  @RequestBody对象校验
     * @param e
     * @return
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public BaseResponse methodArgumentNotValidExceptionHandler(MethodArgumentNotValidException e){
        BindingResult bindingResult = e.getBindingResult();
//        Map<String, String> errors = new HashMap<>();
//        for (FieldError fieldError : bindingResult.getFieldErrors()) {
//            errors.put(fieldError.getField(), fieldError.getDefaultMessage());
//        }
        String msg = bindingResult.getFieldErrors().get(0).getDefaultMessage();
        //集中处理
        //请求参数为空校验
        log.error("MethodArgumentNotValidException：{}",e);
        return ResultUtils.error(ErrorCode.NULL_ERROR.getCode(),msg,msg);
    }

    /**
     * @Validated校验器触发的异常  RequestParam 参数校验
     * @param e
     * @return
     */
    @ExceptionHandler(ConstraintViolationException.class)
    public BaseResponse constraintViolationExceptionHandler(ConstraintViolationException e){
        String msg = e.getConstraintViolations().stream().collect(Collectors.toList()).get(0).getMessage();

        log.error("ConstraintViolationException：{}",e);
        //请求参数为空校验
        return ResultUtils.error(ErrorCode.NULL_ERROR.getCode(),msg,msg);
    }
}
