package com.yd.gateway.service;

import com.alibaba.fastjson.JSON;
import com.alibaba.nacos.api.NacosFactory;
import com.alibaba.nacos.api.config.ConfigService;
import com.alibaba.nacos.api.config.listener.Listener;
import com.alibaba.nacos.api.exception.NacosException;
import com.yd.common.constant.CacheConstants;
import com.yd.common.core.redis.RedisService;
import com.yd.common.utils.StringUtils;
import com.yd.gateway.config.RedisRouteDefinitionRepository;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.gateway.event.RefreshRoutesEvent;
import org.springframework.cloud.gateway.route.RouteDefinition;
import org.springframework.cloud.gateway.support.NotFoundException;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.Executor;

/**
 * 网关动态路由服务
 * 核心功能：通过监听Nacos配置中心，实现网关路由规则的动态更新
 *
 * 工作流程：
 * 1. 初始化时从Nacos加载路由配置
 * 2. 注册Nacos监听器实时接收配置变更
 * 3. 将路由规则持久化到Redis
 * 4. 通过Spring事件机制刷新网关路由
 */
@Service
@Slf4j
public class GatewayDynamicRouteService implements ApplicationEventPublisherAware {

    @Resource
    private RedisRouteDefinitionRepository redisRouteDefinitionRepository; // 路由存储仓库
    @Resource
    private RedisService redisService; // Redis操作服务

    private ApplicationEventPublisher applicationEventPublisher; // Spring事件发布器

    private ConfigService configService; // Nacos配置服务客户端

    private static final long timeout = 30000L; // Nacos配置读取超时时间(30秒)

    // Nacos配置参数
    @Value("${nacos.gateway.route.config.data-id:gateway-router}")
    private String dataId; // 路由配置的dataId(默认gateway-router)
    @Value("${nacos.gateway.route.config.group:DEFAULT_GROUP}")
    private String routeGroup; // 路由配置的分组(默认DEFAULT_GROUP)
    @Value("${spring.cloud.nacos.discovery.server-addr}")
    private String serverAddress; // Nacos服务器地址
    @Value("${spring.cloud.nacos.discovery.namespace:}")
    private String namespace; // Nacos命名空间(可选)

    // 设置Spring事件发布器
    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher applicationEventPublisher) {
        this.applicationEventPublisher = applicationEventPublisher;
    }

    /**
     * 初始化方法 - 服务启动时执行
     * 1. 清空旧路由缓存
     * 2. 初始化Nacos配置服务
     * 3. 加载初始路由配置
     * 4. 注册配置变更监听器
     */
    @PostConstruct
    public void init() {
        log.info("网关路由初始化开始...");

        // 步骤1: 清空Redis中的旧路由缓存
        redisService.deleteObject(CacheConstants.GATEWAY_ROUTES);

        try {
            // 步骤2: 初始化Nacos配置服务客户端
            configService = initConfigService();
            if (configService == null) {
                log.warn("Nacos配置服务初始化失败");
                return;
            }

            // 步骤3: 从Nacos获取初始路由配置
            String configInfo = configService.getConfig(dataId, routeGroup, timeout);
            log.info("从Nacos获取网关路由配置:\r\n{}", configInfo);

            // 步骤4: 解析JSON配置为路由定义对象
            List<RouteDefinition> definitionList = JSON.parseArray(configInfo, RouteDefinition.class);
            for (RouteDefinition definition : definitionList) {
                log.info("添加路由: {}", definition.toString());
                add(definition); // 添加到网关
            }
        } catch (Exception e) {
            log.error("网关路由初始化过程中发生错误", e);
        }

        // 步骤5: 注册Nacos配置变更监听器
        dynamicRouteByNacosListener(dataId, routeGroup);
    }

    /**
     * 初始化Nacos配置服务
     * @return ConfigService实例
     */
    private ConfigService initConfigService() {
        try {
            Properties properties = new Properties();
            properties.setProperty("serverAddr", serverAddress); // 设置Nacos服务器地址

            // 如果配置了命名空间，则添加命名空间参数
            if (StringUtils.isNotBlank(namespace)) {
                properties.setProperty("namespace", namespace);
            }

            // 创建Nacos配置服务实例
            return NacosFactory.createConfigService(properties);
        } catch (Exception e) {
            log.error("创建Nacos配置服务失败", e);
            return null;
        }
    }

    /**
     * 注册Nacos配置变更监听器
     * @param dataId 配置ID
     * @param group 配置分组
     */
    public void dynamicRouteByNacosListener(String dataId, String group) {
        try {
            // 添加Nacos配置监听器
            configService.addListener(dataId, group, new Listener() {
                /**
                 * 当配置变更时的回调方法
                 * @param configInfo 变更后的新配置内容
                 */
                @Override
                public void receiveConfigInfo(String configInfo) {
                    log.info("接收到网关路由配置变更通知:\n\r{}", configInfo);

                    // 解析新配置为路由定义列表
                    List<RouteDefinition> definitionList = JSON.parseArray(configInfo, RouteDefinition.class);
                    log.info("更新路由: {}", definitionList);

                    // 遍历更新每个路由规则
                    definitionList.forEach(route -> update(route));
                }

                /**
                 * 获取监听器执行器
                 * @return 执行器实例(返回null使用默认线程池)
                 */
                @Override
                public Executor getExecutor() {
                    log.info("获取Nacos监听器执行器");
                    return null; // 使用默认线程池
                }
            });
        } catch (NacosException e) {
            log.error("注册Nacos配置监听器失败", e);
        }
    }

    /**
     * 获取所有路由定义
     * @return 路由定义流
     */
    public Flux<RouteDefinition> list() {
        return redisRouteDefinitionRepository.getRouteDefinitions();
    }

    /**
     * 添加新路由
     * @param routeDefinition 路由定义对象
     * @return 操作结果(1=成功)
     */
    public int add(RouteDefinition routeDefinition) {
        // 1. 保存路由到Redis
        redisRouteDefinitionRepository.save(Mono.just(routeDefinition)).subscribe();

        // 2. 发布路由刷新事件
        applicationEventPublisher.publishEvent(new RefreshRoutesEvent(this));

        return 1; // 返回成功标识
    }

    /**
     * 更新路由
     * @param routeDefinition 新的路由定义
     * @return 操作结果(1=成功)
     */
    public int update(RouteDefinition routeDefinition) {
        // 1. 先删除旧路由
        redisRouteDefinitionRepository.delete(Mono.just(routeDefinition.getId()));

        // 2. 再添加新路由
        return add(routeDefinition);
    }

    /**
     * 删除路由
     * @param id 路由ID
     * @return 响应对象(Mono流式响应)
     */
    public Mono<ResponseEntity<Object>> delete(String id) {
        // 1. 从Redis删除路由
        return redisRouteDefinitionRepository.delete(Mono.just(id))
                // 2. 删除成功返回200 OK
                .then(Mono.defer(() -> Mono.just(ResponseEntity.ok().build())))
                // 3. 处理路由不存在的异常(返回404 Not Found)
                .onErrorResume(t -> t instanceof NotFoundException,
                        t -> Mono.just(ResponseEntity.notFound().build()));
    }
}
