package com.yd.csf.api.service.impl;

import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.api.service.ApiAdditionalService;
import com.yd.csf.api.service.ApiAppointmentCheckService;
import com.yd.csf.api.service.ApiProductPlanService;
import com.yd.csf.feign.dto.appointment.ApiProductPlanAdditionalInfoDto;
import com.yd.csf.feign.dto.appointment.ApiProductPlanInfoDto;
import com.yd.csf.feign.dto.appointment.ApiProductPlanMainInfoDto;
import com.yd.csf.service.model.ProductPlan;
import com.yd.csf.service.service.IProductPlanService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;

/**
 * 产品计划业务实现类
 */
@Slf4j
@Service
public class ApiProductPlanServiceImpl implements ApiProductPlanService {

    @Autowired
    private IProductPlanService iProductPlanService;

    @Autowired
    private ApiAppointmentCheckService apiAppointmentCheckService;

    @Autowired
    private ApiAdditionalService apiAdditionalService;

    /**
     * 产品计划详情
     * @param planBizId
     * @return
     */
    @Override
    public Result<ApiProductPlanInfoDto> detail(String appointmentBizId,String planBizId) {
        //产品计划主体详情
        ApiProductPlanInfoDto apiProductPlanInfoDto = new ApiProductPlanInfoDto();
        Result<ApiProductPlanMainInfoDto> result = mainDetail(appointmentBizId,planBizId);
        apiProductPlanInfoDto.setApiProductPlanMainInfoDto(result.getData());
        //附加险列表
        if (!Objects.isNull(result.getData())) {
            Result<List<ApiProductPlanAdditionalInfoDto>> result1 = apiAdditionalService.list(result.getData().getPlanBizId());
            apiProductPlanInfoDto.setApiProductPlanAdditionalInfoDtoList(result1.getData());
        }
        return Result.success(apiProductPlanInfoDto);
    }

    /**
     * 产品计划主体详情
     * @param planBizId
     * @return
     */
    @Override
    public Result<ApiProductPlanMainInfoDto> mainDetail(String appointmentBizId,String planBizId) {
        ProductPlan productPlan = iProductPlanService.queryOne(appointmentBizId,planBizId);
        ApiProductPlanMainInfoDto dto = null;
        if (!Objects.isNull(productPlan)) {
            dto = new ApiProductPlanMainInfoDto();
            BeanUtils.copyProperties(productPlan,dto);
            dto.setProductLaunchMainName(productPlan.getProductLaunchName());
        }
        return Result.success(dto);
    }

    /**
     * 编辑产品计划主体信息
     * @param apiProductPlanMainInfoDto
     * @return
     */
    @Override
    public Result<String> edit(ApiProductPlanMainInfoDto apiProductPlanMainInfoDto) {

        //编辑预约入参字段校验 - 产品计划信息字段校验 - 产品计划主信息字段校验
        apiAppointmentCheckService.checkEditApiProductPlanMainInfoDto(apiProductPlanMainInfoDto);

        ApiProductPlanInfoDto productPlanInfoDto = new ApiProductPlanInfoDto();
        productPlanInfoDto.setApiProductPlanMainInfoDto(apiProductPlanMainInfoDto);
//        Result<ProductPlan> result = editProductPlanData(productPlanInfoDto,apiProductPlanMainInfoDto.getAppointmentBizId());
        String planBizId = "";
//        if (!Objects.isNull(result.getData())) {
//            planBizId = result.getData().getPlanBizId();
//        }
        return Result.success(planBizId);
    }

    /**
     * 编辑产品计划信息
     * @param apiProductPlanInfoDto
     * @return
     */
    @Override
    public Result editPlan(ApiProductPlanInfoDto apiProductPlanInfoDto) {
        //编辑产品计划主体信息
        Result<String> result = edit(apiProductPlanInfoDto.getApiProductPlanMainInfoDto());
        //批量编辑产品计划-附加险信息表数据
        apiAdditionalService.batchEditAdditionalData(apiProductPlanInfoDto,result.getData());
        return Result.success();
    }

    /**
     * 添加产品计划信息表数据
     * @param productPlanInfoDto
     * @param appointmentBizId
     * @return
     */
    public Result<ProductPlan> addProductPlanData(ApiProductPlanInfoDto productPlanInfoDto,String appointmentBizId) {
        if (Objects.isNull(productPlanInfoDto)) {
            //为空设置，方便新建暂存公用方法
            productPlanInfoDto = new ApiProductPlanInfoDto();
        }
        if (Objects.isNull(productPlanInfoDto.getApiProductPlanMainInfoDto())){
            //为空设置，方便新建暂存公用方法
            productPlanInfoDto.setApiProductPlanMainInfoDto(new ApiProductPlanMainInfoDto());
        }
        //产品计划主信息
        ApiProductPlanMainInfoDto dto = productPlanInfoDto.getApiProductPlanMainInfoDto();
        ProductPlan productPlan = new ProductPlan();
        BeanUtils.copyProperties(dto,productPlan);
        //生成产品计划信息表唯一业务ID
        productPlan.setPlanBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_PRODUCT_PLAN.getCode()));
        //预约信息主表唯一业务ID
        productPlan.setAppointmentBizId(appointmentBizId);
        productPlan.setProductLaunchName(dto.getProductLaunchMainName());
        iProductPlanService.saveOrUpdate(productPlan);
        return Result.success(productPlan);
    }

    /**
     * 编辑产品计划信息表数据
     * @param productPlanInfoDto
     * @param appointmentBizId
     * @return
     */
    @Override
    public Result<ProductPlan> editProductPlanData(ApiProductPlanInfoDto productPlanInfoDto, String appointmentBizId) {
        if (Objects.isNull(productPlanInfoDto)
                || ( !Objects.isNull(productPlanInfoDto) && Objects.isNull(productPlanInfoDto.getApiProductPlanMainInfoDto()))){
            //不能为空
            throw new BusinessException("产品计划主信息对象不能为空");
        }
        if (!Objects.isNull(productPlanInfoDto)
                && !Objects.isNull(productPlanInfoDto.getApiProductPlanMainInfoDto())
                && StringUtils.isBlank(productPlanInfoDto.getApiProductPlanMainInfoDto().getPlanBizId())){
            //不能为空
            throw new BusinessException("产品计划信息表唯一业务ID不能为空");
        }
        //产品计划主信息
        ApiProductPlanMainInfoDto dto = productPlanInfoDto.getApiProductPlanMainInfoDto();
        Result<ProductPlan> result = checkProductPlanIsExist(dto.getPlanBizId());
        ProductPlan productPlan = result.getData();
        BeanUtils.copyProperties(dto,productPlan);
        //预约信息主表唯一业务ID
        productPlan.setAppointmentBizId(appointmentBizId);
        productPlan.setProductLaunchName(dto.getProductLaunchMainName());
        iProductPlanService.saveOrUpdate(productPlan);
        return Result.success(productPlan);
    }

    /**
     * 校验产品计划信息是否存在
     * @param planBizId
     * @return
     */
    public Result<ProductPlan> checkProductPlanIsExist(String planBizId) {
        ProductPlan productPlan = iProductPlanService.queryOne(planBizId);
        if (Objects.isNull(productPlan)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(productPlan);
    }
}
