package com.yd.csf.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.constant.CodeGeneratorConstants;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.CodeGenerator;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.api.service.ApiCommissionRuleConfigService;
import com.yd.csf.feign.request.commissionruleconfig.ApiCommissionRuleConfigAddRequest;
import com.yd.csf.feign.request.commissionruleconfig.ApiCommissionRuleConfigEditRequest;
import com.yd.csf.feign.request.commissionruleconfig.ApiCommissionRuleConfigPageRequest;
import com.yd.csf.feign.response.commissionruleconfig.ApiCommissionRuleConfigDetailResponse;
import com.yd.csf.feign.response.commissionruleconfig.ApiCommissionRuleConfigPageResponse;
import com.yd.csf.service.dto.CommissionRuleConfigDto;
import com.yd.csf.service.model.CommissionRuleConfig;
import com.yd.csf.service.service.ICommissionRuleConfigService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;

@Service
public class ApiCommissionRuleConfigServiceImpl implements ApiCommissionRuleConfigService {

    @Autowired
    private ICommissionRuleConfigService iCommissionRuleConfigService;

    /**
     * 分页查询-基本法配置信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiCommissionRuleConfigPageResponse>> page(ApiCommissionRuleConfigPageRequest request) {
        Page<ApiCommissionRuleConfigPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiCommissionRuleConfigPageResponse> iPage = iCommissionRuleConfigService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 详情-基本法配置信息
     * @param ruleBizId
     * @return
     */
    @Override
    public Result<ApiCommissionRuleConfigDetailResponse> detail(String ruleBizId) {
        //校验基本法配置信息是否存在
        Result<CommissionRuleConfig> result = checkCommissionRuleConfigIsExist(ruleBizId);
        CommissionRuleConfig config = result.getData();
        ApiCommissionRuleConfigDetailResponse response = new ApiCommissionRuleConfigDetailResponse();
        BeanUtils.copyProperties(config,response);
        return Result.success(response);
    }

    /**
     * 添加-基本法配置信息
     * @param request
     * @return
     */
    @Override
    public Result add(ApiCommissionRuleConfigAddRequest request) {
        //校验基本法名称的唯一性
        List<CommissionRuleConfig> list = iCommissionRuleConfigService.queryList(CommissionRuleConfigDto.builder()
                .ruleName(request.getRuleName())
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("基本法名称已存在");
        }
        CommissionRuleConfig config = new CommissionRuleConfig();
        BeanUtils.copyProperties(request,config);
        config.setRuleBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_COMMISSION_RULE_CONFIG.getCode()));
        //基本法编码
        config.setRuleCode(CodeGenerator.generate10Code(CodeGeneratorConstants.PREFIX_RULE_CODE));
        iCommissionRuleConfigService.saveOrUpdate(config);
        return Result.success();
    }

    /**
     * 编辑-基本法配置信息
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiCommissionRuleConfigEditRequest request) {
        //校验基本法配置信息是否存在
        Result<CommissionRuleConfig> result = checkCommissionRuleConfigIsExist(request.getRuleBizId());
        CommissionRuleConfig config = result.getData();

        //校验基本法名称的唯一性
        List<CommissionRuleConfig> list = iCommissionRuleConfigService.queryList(CommissionRuleConfigDto.builder()
                .ruleName(request.getRuleName())
                .isExcludeMy(true)
                .ruleBizId(request.getRuleBizId())
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("基本法名称已存在");
        }
        BeanUtils.copyProperties(request,config);
        iCommissionRuleConfigService.saveOrUpdate(config);
        return Result.success();
    }

    /**
     * 删除-基本法配置信息
     * @param ruleBizId
     * @return
     */
    @Override
    public Result del(String ruleBizId) {
        return null;
    }

    /**
     * 校验基本法配置信息是否存在
     * @param ruleBizId
     * @return
     */
    public Result<CommissionRuleConfig> checkCommissionRuleConfigIsExist(String ruleBizId) {
        CommissionRuleConfig config = iCommissionRuleConfigService.queryOne(ruleBizId);
        if (Objects.isNull(config)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(config);
    }
}
