package com.yd.csf.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.api.service.ApiPolicyReceiptService;
import com.yd.csf.feign.request.policyreceipt.ApiPolicyReceiptAddRequest;
import com.yd.csf.feign.request.policyreceipt.ApiPolicyReceiptEditRequest;
import com.yd.csf.feign.request.policyreceipt.ApiPolicyReceiptPageRequest;
import com.yd.csf.feign.response.policyreceipt.ApiPolicyReceiptDetailResponse;
import com.yd.csf.feign.response.policyreceipt.ApiPolicyReceiptPageResponse;
import com.yd.csf.service.model.Policy;
import com.yd.csf.service.model.PolicyReceipt;
import com.yd.csf.service.service.IPolicyReceiptService;
import com.yd.csf.service.service.PolicyService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Objects;

@Slf4j
@Service
public class ApiPolicyReceiptServiceImpl implements ApiPolicyReceiptService {

    @Autowired
    private IPolicyReceiptService iPolicyReceiptService;

    @Autowired
    private PolicyService policyService;

    /**
     * 分页列表查询-保单回执信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiPolicyReceiptPageResponse>> page(ApiPolicyReceiptPageRequest request) {
        Page<ApiPolicyReceiptPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiPolicyReceiptPageResponse> iPage = iPolicyReceiptService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 详情-保单回执信息
     * @param policyReceiptBizId 保单回执表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiPolicyReceiptDetailResponse> detail(String policyReceiptBizId) {
        PolicyReceipt policyReceipt = iPolicyReceiptService.queryOne(policyReceiptBizId);
        if (Objects.isNull(policyReceipt)) {
            throw new BusinessException("回执信息不存在");
        }
        //校验保单信息是否存在
        Policy policy = policyService.queryOne(policyReceipt.getPolicyNo());
        if (Objects.isNull(policy)) {
            throw new BusinessException("保单信息不存在");
        }
        ApiPolicyReceiptDetailResponse response = new ApiPolicyReceiptDetailResponse();
        BeanUtils.copyProperties(policy,response);
        BeanUtils.copyProperties(policyReceipt,response);
        response.setPolicyHolderSignatureList(StringUtils.isNotBlank(policyReceipt.getPolicyHolderSignature()) ? Arrays.asList(policyReceipt.getPolicyHolderSignature().split(";").clone()) : new ArrayList<>());
        response.setInsurer(policy.getInsuranceCompany());
        return Result.success(response);
    }

    /**
     * 添加-保单回执信息
     * @param request
     * @return
     */
    @Override
    public Result add(ApiPolicyReceiptAddRequest request) {
        //校验保单信息是否存在
        Policy policy = policyService.queryOne(request.getPolicyNo());
        if (Objects.isNull(policy)) {
            throw new BusinessException("保单信息不存在");
        }
        PolicyReceipt policyReceipt = new PolicyReceipt();
        policyReceipt.setPolicyNo(request.getPolicyNo());
        policyReceipt.setReceiptDate(request.getReceiptDate());
        policyReceipt.setPolicyReceiptBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_POLICY_RECEIPT.getCode()));
        policyReceipt.setReceiptStatus(request.getReceiptStatus());
        policyReceipt.setPolicyHolderSignature(!CollectionUtils.isEmpty(request.getPolicyHolderSignatureList()) ? String.join(";", request.getPolicyHolderSignatureList()) : "");
        iPolicyReceiptService.saveOrUpdate(policyReceipt);
        return Result.success();
    }

    /**
     * 编辑-保单回执信息
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiPolicyReceiptEditRequest request) {
        //校验保单回执信息是否存在
        PolicyReceipt policyReceipt = iPolicyReceiptService.queryOne(request.getPolicyReceiptBizId());
        if (Objects.isNull(policyReceipt)) {
            throw new BusinessException("保单回执信息不存在");
        }
        //校验保单信息是否存在
        Policy policy = policyService.queryOne(request.getPolicyNo());
        if (Objects.isNull(policy)) {
            throw new BusinessException("保单信息不存在");
        }
        policyReceipt.setPolicyNo(request.getPolicyNo());
        policyReceipt.setReceiptDate(request.getReceiptDate());
        policyReceipt.setReceiptStatus(request.getReceiptStatus());
        policyReceipt.setPolicyHolderSignature(!CollectionUtils.isEmpty(request.getPolicyHolderSignatureList()) ? String.join(";", request.getPolicyHolderSignatureList()) : "");
        iPolicyReceiptService.saveOrUpdate(policyReceipt);
        return Result.success();
    }

    /**
     * 编辑-回执状态
     * @param policyReceiptBizId 保单回执信息表唯一标识(业务ID)
     * @param receiptStatus 回执状态（CONFIRMED-已确认 UNCONFIRMED-未确认）
     * @return
     */
    @Override
    public Result editReceiptStatus(String policyReceiptBizId, String receiptStatus) {
        PolicyReceipt policyReceipt = iPolicyReceiptService.queryOne(policyReceiptBizId);
        if (Objects.isNull(policyReceipt)) {
            throw new BusinessException("回执信息不存在");
        }
        policyReceipt.setReceiptStatus(receiptStatus);
        iPolicyReceiptService.saveOrUpdate(policyReceipt);
        return Result.success();
    }

}
