package com.yd.csf.service.service.impl;

import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import cn.hutool.core.collection.CollUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.service.dto.CommissionQueryRequest;
import com.yd.csf.service.dto.CommissionUpdateRequest;
import com.yd.csf.service.dto.GenerateFortuneRequest;
import com.yd.csf.service.enums.CommissionStatusEnum;
import com.yd.csf.service.enums.FortuneStatusEnum;
import com.yd.csf.service.model.*;
import com.yd.csf.service.service.CommissionService;
import com.yd.csf.service.dao.CommissionMapper;
import com.yd.csf.service.service.FortuneService;
import com.yd.csf.service.service.PolicyBrokerService;
import com.yd.csf.service.service.PolicyService;
import com.yd.csf.service.vo.CommissionVO;
import com.yd.csf.service.vo.FnaVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
* @author Zhang Jianan
* @description 针对表【commission(保单来佣表)】的数据库操作Service实现
* @createDate 2025-09-19 16:08:05
*/
@Service
public class CommissionServiceImpl extends ServiceImpl<CommissionMapper, Commission>
    implements CommissionService{

    @Resource
    private FortuneService fortuneService;

    @Resource
    private PolicyService policyService;

    @Resource
    private PolicyBrokerService policyBrokerService;

    @Override
    public QueryWrapper<Commission> getQueryWrapper(CommissionQueryRequest commissionQueryRequest) {
        QueryWrapper<Commission> queryWrapper = new QueryWrapper<>();
        if (commissionQueryRequest == null) {
            return queryWrapper;
        }

        String status = commissionQueryRequest.getStatus();
        String policyNo = commissionQueryRequest.getPolicyNo();
        String reconciliationCompany = commissionQueryRequest.getReconciliationCompany();
        Date commissionDateStart = commissionQueryRequest.getCommissionDateStart();
        Date commissionDateEnd = commissionQueryRequest.getCommissionDateEnd();

        queryWrapper.eq(StringUtils.isNotBlank(status), "status", status);
        queryWrapper.like(StringUtils.isNotBlank(policyNo), "policy_no", policyNo);
        queryWrapper.like(StringUtils.isNotBlank(reconciliationCompany), "reconciliation_company", reconciliationCompany);

        if (commissionDateStart != null && commissionDateEnd != null) {
            queryWrapper.between("commission_date", commissionDateStart, commissionDateEnd);
        }

        queryWrapper.orderByDesc("id");
        return queryWrapper;
    }

    @Override
    public Page<CommissionVO> getCommissionVOPage(Page<Commission> commissionPage) {
        List<Commission> commissionList = commissionPage.getRecords();
        Page<CommissionVO> commissionVOPage = new Page<>(commissionPage.getCurrent(), commissionPage.getSize(), commissionPage.getTotal());
        if (CollUtil.isEmpty(commissionList)) {
            return commissionVOPage;
        }

        List<CommissionVO> commissionVOList = commissionList.stream().map(CommissionVO::objToVo).collect(Collectors.toList());

        commissionVOPage.setRecords(commissionVOList);
        return commissionVOPage;
    }

    @Override
    public Boolean updateCommission(CommissionUpdateRequest commissionUpdateRequest) {
        String commissionBizId = commissionUpdateRequest.getCommissionBizId();
        Commission commission = this.getOne(new QueryWrapper<Commission>().eq("commission_biz_id", commissionBizId));
        if (commission == null) {
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(), "未找到该来佣记录");
        }
        BeanUtils.copyProperties(commissionUpdateRequest, commission);
        // 获取当前登录用户
        AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
        String loginUserId = currentLoginUser.getId().toString();
        commission.setUpdaterId(loginUserId);
        commission.setUpdateTime(new Date());

        return this.updateById(commission);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean generateFortune(GenerateFortuneRequest generateFortuneRequest) {
        List<String> commissionBizIdList = generateFortuneRequest.getCommissionBizIdList();
        List<Commission> commissionList = this.list(new QueryWrapper<Commission>().in("commission_biz_id", commissionBizIdList));

        // 获取所有保单号
        Set<String> policyNoSet = commissionList.stream().map(Commission::getPolicyNo).collect(Collectors.toSet());

        // 根据保单号查询所有发佣记录
        List<Fortune> fortuneList = fortuneService.list(new QueryWrapper<Fortune>().in("policy_no", policyNoSet));

        // 校验，如果有"可发放或已发并关账"的发佣记录，不能重复生成
        for (Fortune item : fortuneList) {
            if (StringUtils.equals(item.getStatus(), FortuneStatusEnum.CANSEND.getItemValue()) ||
                    StringUtils.equals(item.getStatus(), FortuneStatusEnum.SENT.getItemValue())) {
                throw new BusinessException(ResultCode.FAIL.getCode(), "该来佣记录已被处理，不能重复生成");
            }
        }

        // 1.查询所有保单号对应的保单
        List<Policy> policyList = policyService.list(new QueryWrapper<Policy>().in("policy_no", policyNoSet));
        // 构建保单号到保单的映射
        Map<String, Policy> policyMap = policyList.stream().collect(Collectors.toMap(Policy::getPolicyNo, policy -> policy));

        // 2.根据保单号查询所有转介人
        List<PolicyBroker> brokerList = policyBrokerService.list(new QueryWrapper<PolicyBroker>().in("policy_no", policyNoSet));
        // 构建保单号到转介人的映射
        Map<String, List<PolicyBroker>> brokerMap = brokerList.stream().collect(Collectors.groupingBy(PolicyBroker::getPolicyNo));

        // 3.删除旧的发佣记录
        fortuneService.remove(new QueryWrapper<Fortune>().in("policy_no", policyNoSet));

        // 4. 构建发佣记录
        List<Fortune> newFortuneList = new ArrayList<>();
        for (Commission commission : commissionList) {
            String policyNo = commission.getPolicyNo();

            List<PolicyBroker> policyBrokers = brokerMap.get(policyNo);
            for (PolicyBroker policyBroker : policyBrokers) {
                Fortune fortune = new Fortune();
                fortune.setFortuneBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_FORTUNE.getCode()));
                fortune.setPolicyNo(policyNo);
                fortune.setBroker(policyBroker.getBrokerName());
                fortune.setStatus(FortuneStatusEnum.IN_PROCESS.getItemValue());

                fortune.setCreateTime(new Date());
                fortune.setUpdateTime(new Date());

                newFortuneList.add(fortune);
            }
        }
        // 5. 保存发佣记录
        if (CollUtil.isNotEmpty(newFortuneList)) {
            boolean saveSuccess = fortuneService.saveBatch(newFortuneList);
            if (!saveSuccess) {
                throw new BusinessException(ResultCode.FAIL.getCode(), "保存发佣记录失败");
            }
        }
        return true;
    }
}




