package com.yd.csf.api.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;

import com.yd.csf.api.service.ApiAppointmentService;
import com.yd.csf.feign.dto.appointment.ApiAppointmentInfoDto;
import com.yd.csf.service.dto.*;
import com.yd.csf.service.model.Fna;
import com.yd.csf.service.service.FnaService;
import com.yd.csf.service.vo.FnaVO;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.apache.commons.lang3.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.Map;

/**
 * Fna接口
 *
 * @author jianan
 * @since 2025-08-31
 */
@RestController
@RequestMapping("/Fna")
@Tag(name = "流程接口")
public class ApiFnaController {

    @Resource
    private FnaService fnaService;

    @Resource
    private ApiAppointmentService appointmentService;

    /**
     * 新建流程
     *
     * @param fnaAddRequest
     * @param request
     * @return
     */
    @PostMapping("/add")
    @Operation(summary = "新建流程")
    public Result<Map<String, Object>> addFna(@RequestBody FnaAddRequest fnaAddRequest, HttpServletRequest request) {
        if (fnaAddRequest == null) {
            return Result.fail(ResultCode.PARAMS_ERROR.getCode(), ResultCode.PARAMS_ERROR.getMessage());
        }
        return Result.success(fnaService.addFna(fnaAddRequest));
    }

    /**
     * 删除流程
     *
     * @param deleteRequest
     * @param request
     * @return
     */
    @PostMapping("/delete")
    @Operation(summary = "删除流程")
    public Result<Boolean> deleteFna(@RequestBody DeleteFnaRequest deleteRequest, HttpServletRequest request) {
        if (deleteRequest == null || StringUtils.isBlank(deleteRequest.getFnaBizId())) {
            return Result.fail(ResultCode.PARAMS_ERROR.getCode(), ResultCode.PARAMS_ERROR.getMessage());
        }
        return Result.success(fnaService.deleteFna(deleteRequest));
    }

    /**
     * 更新流程
     *
     * @param fnaUpdateRequest
     * @return
     */
    @PostMapping("/update")
    @Operation(summary = "更新流程")
    public Result<Boolean> updateFna(@RequestBody FnaUpdateRequest fnaUpdateRequest) {
        if (fnaUpdateRequest == null || StringUtils.isBlank(fnaUpdateRequest.getFnaBizId())) {
            throw new BusinessException(ResultCode.PARAMS_ERROR.getCode(), "fnaBizId不能为空");
        }
        String appointmentNo = null;
        if (StringUtils.isNotBlank(fnaUpdateRequest.getAppointmentBizId())) {
            Result<ApiAppointmentInfoDto> apiAppointmentInfoDtoResult = appointmentService.mainDetail(fnaUpdateRequest.getAppointmentBizId());
            if (apiAppointmentInfoDtoResult.getData() != null) {
                ApiAppointmentInfoDto apiAppointmentInfoDto = apiAppointmentInfoDtoResult.getData();
                appointmentNo = apiAppointmentInfoDto.getAppointmentNo();
            }
        }

        return Result.success(fnaService.updateFna(fnaUpdateRequest, appointmentNo));
    }

    /**
     * 根据 fnaBizId 获取fna（封装类）
     *
     * @param fnaBizId
     * @return
     */
    @GetMapping("/get/vo")
    @Operation(summary = "流程详情")
    public Result<FnaVO> getFnaVOByBizId(String fnaBizId, HttpServletRequest request) {
        if (StringUtils.isBlank(fnaBizId)) {
            return Result.fail(ResultCode.PARAMS_ERROR.getCode(), "fnaBizId不能为空");
        }
        // 查询数据库
        Fna fna = fnaService.getByBizId(fnaBizId);
        if (fna == null) {
            return Result.fail(ResultCode.NULL_ERROR.getCode(), ResultCode.NULL_ERROR.getMessage());
        }

        // 获取封装类
        return Result.success(fnaService.getFnaVO(fna));
    }

    /**
     * 分页获取fna列表（仅管理员可用）
     *
     * @param fnaQueryRequest
     * @return
     */
    @PostMapping("/list/page")
    @Operation(summary = "分页获取流程列表")
    public Result<Page<FnaVO>> listFnaByPage(@RequestBody FnaQueryRequest fnaQueryRequest) {
        long current = fnaQueryRequest.getPageNo();
        long size = fnaQueryRequest.getPageSize();
        // 查询数据库
        Page<Fna> fnaPage = fnaService.page(new Page<>(current, size),
                fnaService.getQueryWrapper(fnaQueryRequest));
        return Result.success(fnaService.getFnaVOPage(fnaPage));
    }

    /**
     * 生成副本
     *
     * @param fnaCopyRequest
     * @param request
     * @return
     */
    @PostMapping("/copy")
    @Operation(summary = "生成副本")
    public Result<Map<String, Object>> copyFna(@RequestBody FnaCopyRequest fnaCopyRequest, HttpServletRequest request) {
        if (fnaCopyRequest == null || StringUtils.isBlank(fnaCopyRequest.getFnaBizId())) {
            return Result.fail(ResultCode.PARAMS_ERROR.getCode(), ResultCode.PARAMS_ERROR.getMessage());
        }
        return Result.success(fnaService.copyFna(fnaCopyRequest));
    }
}
