package com.yd.csf.service.service.impl;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.collection.CollUtil;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.google.common.base.Joiner;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.service.dto.*;
import com.yd.csf.service.enums.FortuneStatusEnum;
import com.yd.csf.service.model.*;
import com.yd.csf.service.service.*;
import com.yd.csf.service.dao.FortuneMapper;
import com.yd.csf.service.vo.FortuneVO;
import com.yd.user.feign.client.sysdict.ApiSysDictFeignClient;
import com.yd.user.feign.response.sysdict.GetDictItemListByDictTypeResponse;
import io.swagger.v3.oas.annotations.media.Schema;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.URLEncoder;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Zhang Jianan
 * @description 针对表【fortune(保单发佣表)】的数据库操作Service实现
 * @createDate 2025-09-22 13:22:08
 */
@Service
public class FortuneServiceImpl extends ServiceImpl<FortuneMapper, Fortune>
        implements FortuneService {

    @Resource
    private FortuneAccountService fortuneAccountService;
    @Resource
    private CommissionService commissionService;
    @Resource
    private CommissionExpectedService commissionExpectedService;
    @Resource
    private IExpectedFortuneService expectedFortuneService;
    @Resource
    private PolicyService policyService;
    @Autowired
    private ApiSysDictFeignClient apiSysDictFeignClient;


    @Override
    public Wrapper<Fortune> getQueryWrapper(FortuneQueryRequest fortuneQueryRequest) {
        QueryWrapper<Fortune> queryWrapper = new QueryWrapper<>();
        if (fortuneQueryRequest == null) {
            return queryWrapper;
        }
        String reconciliationYearMonth = fortuneQueryRequest.getReconciliationYearMonth();
        List<String> statusList = fortuneQueryRequest.getStatusList();
        String policyNo = fortuneQueryRequest.getPolicyNo();
        List<String> insuranceCompanyBizIdList = fortuneQueryRequest.getInsuranceCompanyBizIdList();
        List<String> productLaunchBizIdList = fortuneQueryRequest.getProductLaunchBizIdList();
        LocalDate payoutDateStart = fortuneQueryRequest.getPayoutDateStart();
        LocalDate payoutDateEnd = fortuneQueryRequest.getPayoutDateEnd();
        // 入账状态
        List<String> commissionStatusList = fortuneQueryRequest.getCommissionStatusList();

        queryWrapper.eq(StringUtils.isNotBlank(reconciliationYearMonth), "reconciliation_year_month", reconciliationYearMonth);
        queryWrapper.in(CollUtil.isNotEmpty(statusList), "status", statusList);
        queryWrapper.eq(StringUtils.isNotBlank(policyNo), "policy_no", policyNo);
        queryWrapper.in(CollUtil.isNotEmpty(insuranceCompanyBizIdList), "insurance_company_biz_id", insuranceCompanyBizIdList);
        queryWrapper.in(CollUtil.isNotEmpty(productLaunchBizIdList), "product_launch_biz_id", productLaunchBizIdList);

        queryWrapper.gt(ObjectUtils.isNotEmpty(payoutDateStart), "payout_date", payoutDateStart);
        queryWrapper.lt(ObjectUtils.isNotEmpty(payoutDateEnd), "payout_date", payoutDateEnd);

        // 入账状态, 关联查询 commission_expected 表
        if (CollUtil.isNotEmpty(commissionStatusList)) {
            queryWrapper.apply("EXISTS (SELECT 1 FROM commission_expected ce WHERE ce.commission_expected_biz_id = fortune.commission_expected_biz_id AND ce.status IN ({0}))",
                    Joiner.on(",").join(commissionStatusList));
        }

        queryWrapper.orderByAsc("id");
        return queryWrapper;
    }

    @Override
    public Page<FortuneVO> getFortuneVOPage(Page<Fortune> fortunePage) {
        List<Fortune> fortuneList = fortunePage.getRecords();
        Page<FortuneVO> fortuneVOPage = new Page<>(fortunePage.getCurrent(), fortunePage.getSize(), fortunePage.getTotal());
        if (CollUtil.isEmpty(fortuneList)) {
            return fortuneVOPage;
        }

        List<FortuneVO> fortuneVOList = new ArrayList<>();
        Set<String> expectedFortuneBizIdSet = new HashSet<>();
        Set<String> policyNoSet = new HashSet<>();
        for (Fortune fortune : fortuneList) {
            FortuneVO fortuneVO = FortuneVO.objToVo(fortune);
            fortuneVOList.add(fortuneVO);
            expectedFortuneBizIdSet.add(fortune.getExpectedFortuneBizId());
            policyNoSet.add(fortune.getPolicyNo());
        }

        Set<String> commissionExpectedBizIdSet = fortuneList.stream().map(Fortune::getCommissionExpectedBizId).collect(Collectors.toSet());
        // 1 关联查询来佣预期信息
        List<CommissionExpected> commissionExpectedList = commissionExpectedService.lambdaQuery()
                .in(CommissionExpected::getCommissionExpectedBizId, commissionExpectedBizIdSet)
                .select(CommissionExpected::getCommissionExpectedBizId, CommissionExpected::getPaidAmount, CommissionExpected::getPaidRatio)
                .list();
        Map<String, CommissionExpected> commissionExpectedMap = commissionExpectedList.stream().collect(Collectors.toMap(CommissionExpected::getCommissionExpectedBizId, a -> a, (oldValue, newValue) -> newValue));

        // 2.关联查询保单信息
        QueryWrapper<Policy> queryWrapper = new QueryWrapper<Policy>();
        queryWrapper.select("policy_no", "payment_premium", "product_name", "insurance_company", "reconciliation_company");
        queryWrapper.in("policy_no", policyNoSet);
        List<Policy> policyList = policyService.list(queryWrapper);
        Map<String, Policy> policyMap = policyList.stream().collect(Collectors.toMap(Policy::getPolicyNo, a -> a, (oldValue, newValue) -> newValue));

        // 3.关联查询预计出账信息
        List<ExpectedFortune> expectedFortuneList = expectedFortuneService.list(new QueryWrapper<ExpectedFortune>().in("expected_fortune_biz_id", expectedFortuneBizIdSet));
        Map<String, ExpectedFortune> expectedFortuneMap = expectedFortuneList.stream().collect(Collectors.toMap(ExpectedFortune::getExpectedFortuneBizId, a -> a, (oldValue, newValue) -> newValue));

        fortuneVOList.forEach(fortuneVO -> {
            String commissionBizId = fortuneVO.getCommissionExpectedBizId();
            // 设置来佣金额和比例
            CommissionExpected commissionExpected = commissionExpectedMap.get(commissionBizId);
            if (commissionExpected != null) {
                fortuneVO.setCommissionPaidAmount(commissionExpected.getPaidAmount());
                fortuneVO.setCommissionPaidRatio(commissionExpected.getPaidRatio());
            }

            // 设置保单信息
            Policy policy = policyMap.get(fortuneVO.getPolicyNo());
            if (policy != null) {
                fortuneVO.setPremium(policy.getPaymentPremium());
                fortuneVO.setInsuranceCompany(policy.getInsuranceCompany());
                fortuneVO.setProductName(policy.getProductName());
            }
        });

        fortuneVOPage.setRecords(fortuneVOList);
        return fortuneVOPage;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean updateFortune(FortuneUpdateRequest fortuneUpdateRequest) {
        if (fortuneUpdateRequest.getStatus() == null) {
            throw new BusinessException(ResultCode.PARAM_CHECK_ERROR.getCode(), "出账状态不能为空");
        }
        if (!StringUtils.equalsAny(fortuneUpdateRequest.getStatus(), FortuneStatusEnum.RESERVED.getItemValue(), FortuneStatusEnum.INVALID.getItemValue(), FortuneStatusEnum.CHECKED.getItemValue(), FortuneStatusEnum.CAN_SEND.getItemValue())) {
            throw new BusinessException(ResultCode.PARAM_CHECK_ERROR.getCode(), "出账状态只能为可出帐，待检核、保留、已失效、可出帐，检核完成");
        }

        String fortuneBizId = fortuneUpdateRequest.getFortuneBizId();
        // 查询出账数据
        Fortune fortune = this.getOne(new QueryWrapper<Fortune>().eq("fortune_biz_id", fortuneBizId));
        if (fortune == null) {
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(), "发佣数据不存在");
        }
        // 查询预计出账数据
        ExpectedFortune expectedFortune = expectedFortuneService.getOne(new QueryWrapper<ExpectedFortune>().eq("expected_fortune_biz_id", fortune.getExpectedFortuneBizId()));
        if (expectedFortune == null) {
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(), "预计出账数据不存在");
        }
        BigDecimal currentPaymentAmount = fortuneUpdateRequest.getCurrentPaymentAmount();
        if (currentPaymentAmount != null) {
            if (currentPaymentAmount.compareTo(BigDecimal.ZERO) < 0) {
                throw new BusinessException(ResultCode.PARAM_CHECK_ERROR.getCode(), "出账金额不能小于0");
            }
            if (currentPaymentAmount.compareTo(expectedFortune.getAmount()) > 0) {
                throw new BusinessException(ResultCode.PARAM_CHECK_ERROR.getCode(), "出账金额不能大于应付款金额");
            }
            if (StringUtils.isBlank(fortuneUpdateRequest.getCurrency())) {
                throw new BusinessException(ResultCode.PARAM_CHECK_ERROR.getCode(), "出账币种不能为空");
            }
        }
        // 更新结算汇率
        BigDecimal exchangeRate = ObjectUtils.isNotEmpty((fortuneUpdateRequest.getExchangeRate()))
                ? fortuneUpdateRequest.getExchangeRate()
                : queryDefaultExchangeRate(fortuneUpdateRequest.getCurrency());
        fortune.setExchangeRate(exchangeRate);

        if (ObjectUtils.isEmpty(fortune.getHkdAmount())) {
            fortune.setHkdAmount(fortune.getAmount().multiply(exchangeRate));
        }

        // 获取当前登录用户
        AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
        String loginUserId = currentLoginUser.getId().toString();

        // 检核人
        fortune.setReconciliationOperator(currentLoginUser.getUsername());

        // 如果传入了金额
        if (currentPaymentAmount != null) {
            splitFortune(fortune, currentPaymentAmount, expectedFortune, loginUserId, fortuneUpdateRequest);
        } else {
            // 如果未传入金额，仅更新预计出账状态为已出帐
            expectedFortuneService.lambdaUpdate()
                    .set(ExpectedFortune::getStatus, fortuneUpdateRequest.getStatus())
                    .eq(ExpectedFortune::getId, expectedFortune.getId())
                    .update();

            // 统一更新实际出账数据
            BeanUtils.copyProperties(fortuneUpdateRequest, fortune, "id", "fortuneBizId", "expectedFortuneBizId", "commissionBizId");

            // 更新实际出账数据
            fortune.setUpdaterId(loginUserId);
            fortune.setUpdateTime(new Date());

            this.updateById(fortune);
        }
        return true;
    }

    private BigDecimal queryDefaultExchangeRate(String currency) {
        if (StringUtils.isBlank(currency)) {
            return BigDecimal.ONE;
        }
        if ("HKD".equalsIgnoreCase(currency)) {
            return BigDecimal.valueOf(1);
        }
        Result<List<GetDictItemListByDictTypeResponse>> result = apiSysDictFeignClient.getDictItemListByDictType("csf_exchange_rate_hkd");
        if (org.apache.commons.collections4.CollectionUtils.isNotEmpty(result.getData())) {
            for (GetDictItemListByDictTypeResponse dictItem : result.getData()) {
                if (StringUtils.equalsIgnoreCase(dictItem.getItemLabel(), currency)) {
                    return new BigDecimal(dictItem.getItemValue());
                }
            }
        }
        return BigDecimal.ONE;
    }

    /**
     * 拆分发放（按本次发放金额拆）
     */
    @Transactional(rollbackFor = Exception.class)
    public void splitFortune(Fortune main, BigDecimal currentPaymentAmount,ExpectedFortune expectedFortune, String loginUserId, FortuneUpdateRequest fortuneUpdateRequest) {
        if (currentPaymentAmount == null || currentPaymentAmount.compareTo(BigDecimal.ZERO) <= 0)
            throw new BusinessException("本次发放金额必须大于0");

        if (!main.getStatus().equals(FortuneStatusEnum.CAN_SEND.getItemValue()))
            throw new BusinessException("只有可出账记录才能修改金额");

        BigDecimal fullAmount = main.getAmount();
        if (currentPaymentAmount.compareTo(main.getCurrentPaymentAmount()) > 0)
            throw new BusinessException("本次发放金额不能大于可出账金额");

        // 本次发放港币金额
        BigDecimal currentHkdAmount = currentPaymentAmount.multiply(main.getExchangeRate());
        // 1. 更新主行 = 本次发放金额
        main.setCurrentPaymentAmount(currentPaymentAmount);
        main.setCurrentPaymentHkdAmount(currentHkdAmount);
        main.setCurrentPaymentRatio(currentPaymentAmount
                .divide(expectedFortune.getAmount(), 4, RoundingMode.HALF_UP).multiply(BigDecimal.valueOf(100)));
        main.setStatus(FortuneStatusEnum.CHECKED.getItemValue());
        main.setUpdaterId(loginUserId);
        main.setUpdateTime(new Date());
        // 其他字段更新
        if (StringUtils.isNotBlank(fortuneUpdateRequest.getCurrency())) {
            main.setCurrency(fortuneUpdateRequest.getCurrency());
        }
        this.updateById(main);

        // 2. 剩余金额生成新行
        BigDecimal leftAmount = fullAmount.subtract(currentPaymentAmount);
        if (leftAmount.compareTo(BigDecimal.ZERO) > 0) {
            Fortune part = new Fortune();
            BeanUtils.copyProperties(main, part, "id", "fortuneBizId", "currentPaymentAmount", "status", "isPart", "payoutDate");
            part.setFortuneBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_FORTUNE.getCode()));
            part.setCurrentPaymentAmount(leftAmount);
            part.setCurrentPaymentHkdAmount(leftAmount.multiply(main.getExchangeRate()));
            part.setCurrentPaymentRatio(leftAmount
                    .divide(expectedFortune.getAmount(), 4, RoundingMode.HALF_UP).multiply(BigDecimal.valueOf(100)));
            part.setStatus(FortuneStatusEnum.WAIT.getItemValue());
            part.setIsPart(1);
            part.setPayoutDate(LocalDate.now().plusMonths(1)); // 次月
            part.setCreatorId(loginUserId);
            part.setCreateTime(new Date());
            this.save(part);
        }

        // 3. 同步 expected_fortune
        ExpectedFortune ef = expectedFortuneService.lambdaQuery()
                .eq(ExpectedFortune::getExpectedFortuneBizId, main.getExpectedFortuneBizId())
                .one();
        if (ef == null) return;

        // 统一使用港币金额
        BigDecimal newPaid = ef.getPaidAmount() == null ? BigDecimal.ZERO : ef.getPaidAmount().add(currentHkdAmount);
        BigDecimal newUnpaid = ef.getHkdAmount().subtract(newPaid);
        String newStatus = newUnpaid.compareTo(BigDecimal.ZERO) == 0 ? "6" : "3";

        expectedFortuneService.lambdaUpdate()
                .set(ExpectedFortune::getPaidAmount, newPaid)
                .set(ExpectedFortune::getUnpaidAmount, newUnpaid)
                .set(ExpectedFortune::getPaidRatio,
                        newPaid.divide(ef.getHkdAmount(), 4, RoundingMode.HALF_UP).multiply(BigDecimal.valueOf(100)))
                .set(ExpectedFortune::getUnpaidRatio,
                        newUnpaid.divide(ef.getHkdAmount(), 4, RoundingMode.HALF_UP).multiply(BigDecimal.valueOf(100)))
                .set(ExpectedFortune::getStatus, newStatus)
                .eq(ExpectedFortune::getId, ef.getId())
                .update();
    }

    @Override
    public void downloadFortune(FortuneDownloadRequest fortuneDownloadRequest, HttpServletResponse response) throws IOException {
        List<String> fortuneBizIdList = fortuneDownloadRequest.getFortuneBizIdList();
        if (CollUtil.isNotEmpty(fortuneBizIdList)) {
            List<Fortune> fortuneList = this.list(new QueryWrapper<Fortune>().in("fortune_biz_id", fortuneBizIdList));
            if (CollUtil.isNotEmpty(fortuneList)) {
                // 转换为导出DTO
                List<FortuneExportDTO> exportDTOList = fortuneList.stream().map(fortune -> {
                    FortuneExportDTO exportDTO = new FortuneExportDTO();
                    BeanUtils.copyProperties(fortune, exportDTO);
                    return exportDTO;
                }).collect(Collectors.toList());

                // 设置响应头
                response.setContentType("application/vnd.ms-excel;charset=UTF-8");
                response.setCharacterEncoding("utf-8");
                String encode = URLEncoder.encode("保单发佣.xlsx", "UTF-8");
                response.setHeader("Content-Disposition", "attachment;filename=" + "fortune_"+System.currentTimeMillis() + ".xlsx");

                // 导出
                try {
                    EasyExcel.write(response.getOutputStream(), FortuneExportDTO.class).sheet("发佣").doWrite(exportDTOList);
                } catch (IOException e) {
                    log.error("导出保单发佣数据失败", e);
                    throw new BusinessException(ResultCode.FAIL.getCode(), "导出失败");
                }
            }
        }
    }

    @Override
    public void downloadAccount(FortuneDownloadRequest fortuneDownloadRequest, HttpServletResponse response) throws UnsupportedEncodingException {
        List<String> fortuneBizIdList = fortuneDownloadRequest.getFortuneBizIdList();
        if (CollUtil.isNotEmpty(fortuneBizIdList)) {
            List<Fortune> fortuneList = this.list(new QueryWrapper<Fortune>().in("fortune_biz_id", fortuneBizIdList));
            // 校验发佣记录状态
            StringBuilder validateMsg = new StringBuilder();
            for (Fortune fortune : fortuneList) {
                if (StringUtils.equals(fortune.getStatus(), FortuneStatusEnum.CAN_SEND.getItemValue())) {
                    validateMsg.append(fortune.getPolicyNo()).append("-").append(fortune.getBroker()).append("当前未检核，请检核后再生成出账记录; ");
                }
                if (StringUtils.equals(fortune.getStatus(), FortuneStatusEnum.SENT.getItemValue())) {
                    validateMsg.append(fortune.getPolicyNo()).append("-").append(fortune.getBroker()).append("已完成出账，不能生成出账记录; ");
                }
            }
            if (StringUtils.isNotBlank(validateMsg.toString())) {
                throw new BusinessException(ResultCode.FAIL.getCode(), validateMsg.toString());
            }

            if (CollUtil.isNotEmpty(fortuneList)) {
                // 按人和币种分组
                Map<String, List<Fortune>> fortuneMap = fortuneList.stream().collect(Collectors.groupingBy(fortune -> fortune.getBroker() + "_" + fortune.getCurrency()));

                // 创建按人分组的导出DTO列表
                List<FortuneAccountExportDTO> accountExportDTOList = new ArrayList<>();

                // 收集所有不同的fortune项目名称
                Set<String> allFortuneNames = fortuneList.stream()
                        .map(Fortune::getFortuneName)
                        .filter(StringUtils::isNotBlank)
                        .collect(Collectors.toSet());

                // 处理每个分组的数据
                for (Map.Entry<String, List<Fortune>> entry : fortuneMap.entrySet()) {
                    String broker = entry.getKey().split("_")[0];
                    String currency = entry.getKey().split("_")[1];
                    List<Fortune> brokerFortunes = entry.getValue();

                    FortuneAccountExportDTO accountDTO = new FortuneAccountExportDTO();
                    accountDTO.setBroker(broker);
                    accountDTO.setFortuneList(brokerFortunes);

                    // 设置团队、币种（取第一个记录）
                    if (CollUtil.isNotEmpty(brokerFortunes)) {
                        accountDTO.setTeam(brokerFortunes.get(0).getTeam());
                        accountDTO.setCurrency(currency);
                    }

                    // 计算出账总额
                    BigDecimal totalAmount = brokerFortunes.stream()
                            .map(Fortune::getCurrentPaymentAmount)
                            .reduce(BigDecimal.ZERO, BigDecimal::add);
                    accountDTO.setAmount(totalAmount);

                    // 按fortune项目分组并累加金额
                    Map<String, BigDecimal> fortuneAmountMap = brokerFortunes.stream()
                            .collect(Collectors.groupingBy(
                                    Fortune::getFortuneName,
                                    Collectors.reducing(BigDecimal.ZERO, Fortune::getCurrentPaymentAmount, BigDecimal::add)
                            ));

                    // 设置各个fortune项目的金额
                    fortuneAmountMap.forEach(accountDTO::addFortuneAmount);

                    accountExportDTOList.add(accountDTO);
                }

//                // 设置响应头
//                response.setContentType("application/vnd.ms-excel;charset=UTF-8");
//                response.setCharacterEncoding("utf-8");
//                String fileName = "fortune_account_" + System.currentTimeMillis() + ".xlsx";
//                response.setHeader("Content-Disposition", "attachment;filename=" + fileName);
//
//                // 创建动态表头
//                List<List<String>> head = new ArrayList<>();
//                head.add(Collections.singletonList("转介人"));
//                head.add(Collections.singletonList("所属团队"));
//                head.add(Collections.singletonList("出账币种"));
//                head.add(Collections.singletonList("出账总额"));
//
//                // 添加所有fortune项目作为表头
//                for (String fortuneName : allFortuneNames) {
//                    head.add(Collections.singletonList(fortuneName + "金额"));
//                }
//
//                // 构建数据行
//                List<List<Object>> dataList = new ArrayList<>();
//                for (FortuneAccountExportDTO dto : accountExportDTOList) {
//                    List<Object> row = new ArrayList<>();
//                    row.add(dto.getBroker());
//                    row.add(dto.getTeam());
//                    row.add(dto.getCurrency());
//                    row.add(dto.getAmount());
//                    for (String fortuneName : allFortuneNames) {
//                        row.add(dto.getFortuneAmount(fortuneName));
//                    }
//                    dataList.add(row);
//                }

                // 写入数据库
                fortuneAccountService.saveFortuneAccount(accountExportDTOList);

//                // 使用try-with-resources确保流正确关闭
//                try (ServletOutputStream outputStream = response.getOutputStream()) {
//                    // 写入 Excel
//                    EasyExcel.write(outputStream)
//                            .head(head)
//                            .sheet("出账清单")
//                            .doWrite(dataList);
//
//                    outputStream.flush();
//                } catch (Exception e) {
//                    log.error("导出失败", e);
//                    throw new BusinessException(ResultCode.FAIL.getCode(), "导出失败");
//                }
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean updateFortuneStatus(FortuneStatusUpdateRequest fortuneStatusUpdateRequest) {

        List<String> fortuneBizIdList = fortuneStatusUpdateRequest.getFortuneBizIdList();
        List<Fortune> fortuneList = this.list(new QueryWrapper<Fortune>().in("fortune_biz_id", fortuneBizIdList));

        // 获取当前登录用户
        AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
        String loginUserId = currentLoginUser.getId().toString();

        Date now = new Date();
        List<Fortune> updateList = new ArrayList<>();
        for (Fortune fortune : fortuneList) {
            Fortune updateFortune = new Fortune();
            updateFortune.setId(fortune.getId());
            updateFortune.setUpdaterId(loginUserId);
            updateFortune.setUpdateTime(now);
            updateFortune.setStatus(fortuneStatusUpdateRequest.getStatus());
            updateList.add(updateFortune);
        }

        return this.updateBatchById(updateList);
    }

    @Override
    public Fortune getByFortuneBizId(String fortuneBizId) {
        return this.getOne(new QueryWrapper<Fortune>().eq("fortune_biz_id", fortuneBizId));
    }

    @Override
    public Boolean addFortune(FortuneAddRequest fortuneAddRequest) {
        Fortune fortune = new Fortune();
        BeanUtil.copyProperties(fortuneAddRequest, fortune);
        // 生成发佣业务ID
        fortune.setFortuneBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_FORTUNE.getCode()));
        fortune.setCurrentPaymentAmount(fortuneAddRequest.getAmount());
        fortune.setFortuneName(queryByDict(fortuneAddRequest.getFortuneType()));
        fortune.setIsPart(0);
        fortune.setStatus(FortuneStatusEnum.WAIT.getItemValue());

        return this.save(fortune);
    }

    private String queryByDict(String fortuneType) {
        Result<List<GetDictItemListByDictTypeResponse>> result = apiSysDictFeignClient.getDictItemListByDictType("csf_fortune_type");
        if (CollectionUtils.isNotEmpty(result.getData())) {
            for (GetDictItemListByDictTypeResponse dictItem : result.getData()) {
                if (StringUtils.equalsIgnoreCase(dictItem.getItemValue(), fortuneType)) {
                    return dictItem.getItemLabel();
                }
            }
        }
        return null;
    }

    /**
     * 查询单个对象
     * @param fortuneBizId
     * @return
     */
    @Override
    public Fortune queryOne(String fortuneBizId) {
        return this.getOne(new LambdaQueryWrapper<Fortune>().eq(Fortune::getFortuneBizId,fortuneBizId));
    }

    @Override
    public FortuneStatisticsVO getFortuneStatistics(List<Long> fortuneIdList) {
        if (CollectionUtils.isEmpty(fortuneIdList)) {
            FortuneStatisticsVO fortuneStatisticsVO = new FortuneStatisticsVO();
            fortuneStatisticsVO.setTotalInAmount(BigDecimal.ZERO);
            fortuneStatisticsVO.setTotalPolicyCount(0L);
            fortuneStatisticsVO.setTotalPremium(BigDecimal.ZERO);
            fortuneStatisticsVO.setPendingOutAmount(BigDecimal.ZERO);
            fortuneStatisticsVO.setAvailableOutAmount(BigDecimal.ZERO);
            fortuneStatisticsVO.setDifferenceAmount(BigDecimal.ZERO);
            return fortuneStatisticsVO;
        }
        // 自定义查询
        List<FortuneStatisticsVO> fortuneStatisticsList = this.baseMapper.getFortuneStatistics(fortuneIdList);

        // 查询结果
        FortuneStatisticsVO fortuneStatistics;
        if (CollectionUtils.isNotEmpty(fortuneStatisticsList) && fortuneStatisticsList.size() > 1) {
            // 合并查询结果
            FortuneStatisticsVO noPolicyNoData = fortuneStatisticsList.get(0);
            FortuneStatisticsVO policyNoData = fortuneStatisticsList.get(1);
            // 合并查询结果
            fortuneStatistics = new FortuneStatisticsVO();
            fortuneStatistics.setTotalInAmount(policyNoData.getTotalInAmount());
            fortuneStatistics.setTotalPolicyCount(policyNoData.getTotalPolicyCount());
            fortuneStatistics.setTotalPremium(policyNoData.getTotalPremium());
            fortuneStatistics.setPendingOutAmount(noPolicyNoData.getPendingOutAmount().add(policyNoData.getPendingOutAmount()));
            fortuneStatistics.setAvailableOutAmount(noPolicyNoData.getAvailableOutAmount().add(policyNoData.getAvailableOutAmount()));
        } else {
            fortuneStatistics = fortuneStatisticsList.get(0);
        }
        // 计算差额
        fortuneStatistics.setDifferenceAmount(fortuneStatistics.getPendingOutAmount().subtract(fortuneStatistics.getAvailableOutAmount()));
        return fortuneStatistics;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean addFortuneBatch(List<FortuneAddRequest> fortuneAddRequestList) {
        if (CollectionUtils.isEmpty(fortuneAddRequestList)) {
            return false;
        }

        List<Fortune> fortuneList = new ArrayList<>();
        for (FortuneAddRequest fortuneAddRequest : fortuneAddRequestList) {
            Fortune fortune = new Fortune();
            BeanUtil.copyProperties(fortuneAddRequest, fortune);
            // 生成发佣业务ID
            fortune.setFortuneBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_FORTUNE.getCode()));
            fortune.setStatus(FortuneStatusEnum.WAIT.getItemValue());
            fortuneList.add(fortune);
        }
        return this.saveBatch(fortuneList);
    }

}




