package com.yd.csf.service.utils;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.reflect.TypeToken;

import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;

/**
 * GSON工具类
 */
public class GSONUtil {

    // 默认配置的Gson实例
    private static final Gson DEFAULT_GSON = new GsonBuilder()
            .setDateFormat("yyyy-MM-dd HH:mm:ss")
            .disableHtmlEscaping()
            .create();

    // 支持null值的Gson实例
    private static final Gson NULL_SAFE_GSON = new GsonBuilder()
            .setDateFormat("yyyy-MM-dd HH:mm:ss")
            .serializeNulls()
            .disableHtmlEscaping()
            .create();

    // 美化输出的Gson实例
    private static final Gson PRETTY_GSON = new GsonBuilder()
            .setDateFormat("yyyy-MM-dd HH:mm:ss")
            .setPrettyPrinting()
            .disableHtmlEscaping()
            .create();

    /**
     * 对象转JSON字符串（默认配置）
     */
    public static String toJson(Object obj) {
        return DEFAULT_GSON.toJson(obj);
    }

    /**
     * 对象转JSON字符串（包含null值）
     */
    public static String toJsonWithNulls(Object obj) {
        return NULL_SAFE_GSON.toJson(obj);
    }

    /**
     * 对象转格式化的JSON字符串
     */
    public static String toPrettyJson(Object obj) {
        return PRETTY_GSON.toJson(obj);
    }

    /**
     * JSON字符串转对象
     */
    public static <T> T fromJson(String json, Class<T> clazz) {
        return DEFAULT_GSON.fromJson(json, clazz);
    }

    /**
     * JSON字符串转对象（支持泛型）
     */
    public static <T> T fromJson(String json, Type type) {
        return DEFAULT_GSON.fromJson(json, type);
    }

    /**
     * JSON字符串转List
     */
    public static <T> List<T> fromJsonToList(String json, Class<T> clazz) {
        Type type = TypeToken.getParameterized(List.class, clazz).getType();
        return DEFAULT_GSON.fromJson(json, type);
    }

    /**
     * JSON字符串转Map
     */
    public static <K, V> Map<K, V> fromJsonToMap(String json, Class<K> keyClass, Class<V> valueClass) {
        Type type = TypeToken.getParameterized(Map.class, keyClass, valueClass).getType();
        return DEFAULT_GSON.fromJson(json, type);
    }

    /**
     * 获取自定义配置的Gson实例
     */
    public static Gson getCustomGson(GsonBuilder builder) {
        return builder.create();
    }

    /**
     * 向后兼容 - 保留原有的静态实例
     */
    public static Gson gson = DEFAULT_GSON;
}
