package com.yd.csf.api.controller;

import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.result.Result;
import com.yd.csf.api.dto.FortuneImportDTO;
import com.yd.csf.service.common.ErrorCode;
import com.yd.csf.service.dto.FortuneQueryRequest;
import com.yd.csf.service.dto.FortuneUpdateRequest;
import com.yd.csf.service.model.Fortune;
import com.yd.csf.service.service.FortuneService;
import com.yd.csf.service.vo.FortuneVO;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * 保单发佣接口
 *
 * @author jianan
 * @since 2025-09-15
 */
@RestController
@RequestMapping("/fortune")
@Tag(name = "保单发佣接口")
public class ApiFortuneController {

    private static final Logger log = LoggerFactory.getLogger(ApiFortuneController.class);

    @Resource
    private FortuneService fortuneService;


    @PostMapping("/upload/excel")
    @Transactional(rollbackFor = Exception.class)
    public Result<Boolean> uploadExcel(@RequestParam("file") MultipartFile file) throws IOException {
        try {
            List<FortuneImportDTO> dataList = EasyExcel.read(file.getInputStream())
                    .head(FortuneImportDTO.class)
                    .sheet("发佣")
                    .doReadSync();

            // 获取当前登录用户的ID
            AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
            String loginUserId = currentLoginUser.getId().toString();

            // 数据验证和处理
            processData(dataList, loginUserId);

        } catch (IOException e) {
            throw new RuntimeException("Excel读取失败", e);
        }

        return Result.success(true);
    }

    /**
     * 处理导入的数据
     */
    private void processData(List<FortuneImportDTO> dataList, String loginUserId) {
        List<Fortune> entities = new ArrayList<>();
        for (FortuneImportDTO data : dataList) {
            // 数据验证
            Fortune entity = FortuneImportDTO.convertToEntity(data, loginUserId);
            entities.add(entity);
        }
        // 批量保存
        fortuneService.saveBatch(entities);
    }


    /**
     * 创建保单发佣
     *
     * @param customerAddRequest
     * @param request
     * @return
     */
//    @Operation(summary = "创建保单发佣")
//    @PostMapping("/add")
//    @Transactional(rollbackFor = Exception.class)
//    public Result<Map<String, Object>> addCustomer(@RequestBody CustomerAddRequest customerAddRequest, HttpServletRequest request) {
//        if (customerAddRequest == null) {
//            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
//        }
//        return Result.success(policyFollowService.addCustomer(customerAddRequest));
//    }

    /**
     * 删除fna
     *
     * @param deleteRequest
     * @param request
     * @return
     */
//    @PostMapping("/delete")
//    public Result<Boolean> deleteFna(@RequestBody DeleteRequest deleteRequest, HttpServletRequest request) {
//        if (deleteRequest == null || deleteRequest.getId() <= 0) {
//            throw new BusinessException(ErrorCode.PARAMS_ERROR);
//        }
//        User user = userService.getLoginUser(request);
//        long id = deleteRequest.getId();
//        // 判断是否存在
//        Customer oldFna = policyFollowService.getById(id);
//        ThrowUtils.throwIf(oldFna == null, ErrorCode.NOT_FOUND_ERROR);
//        // 仅本人或管理员可删除
//        if (!oldFna.getUserId().equals(user.getId()) && !userService.isAdmin(request)) {
//            throw new BusinessException(ErrorCode.NO_AUTH_ERROR);
//        }
//        // 操作数据库
//        boolean result = policyFollowService.removeById(id);
//        ThrowUtils.throwIf(!result, ErrorCode.OPERATION_ERROR);
//        return Result.success(true);
//    }

    /**
     * 更新保单发佣
     *
     * @param fortuneUpdateRequest
     * @return
     */
    @PostMapping("/update")
    @Operation(summary = "更新保单发佣信息")
    public Result<Boolean> updateFortune(@RequestBody FortuneUpdateRequest fortuneUpdateRequest) {
        if (fortuneUpdateRequest == null || fortuneUpdateRequest.getFortuneBizId() == null) {
            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }
        return Result.success(fortuneService.updateFortune(fortuneUpdateRequest));
    }

    /**
     * 根据 fortuneBizId 获取保单发佣（封装类）
     *
     * @param fortuneBizId
     * @return
     */
//    @GetMapping("/get/vo")
//    @Operation(summary = "根据 policyBizId 获取保单发佣详情")
//    public Result<PolicyFollowVO> getPolicyFollowByPolicyBizId(@RequestParam("fortuneBizId") String fortuneBizId, HttpServletRequest request) {
//        if (fortuneBizId == null) {
//            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
//        }
//        // 查询数据库
//        PolicyFollow policyFollow = policyFollowService.getByPolicyBizId(policyBizId);
//        if (policyFollow == null) {
//            return Result.fail(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
//        }
//
//        // 获取封装类
//        return Result.success(policyFollowService.getPolicyFollowVO(policyFollow));
//    }

    /**
     * 分页获取保单发佣列表（仅管理员可用）
     *
     * @param fnaQueryRequest
     * @return
     */
//    @PostMapping("/list/page")
//    public Result<Page<Customer>> listFnaByPage(@RequestBody FnaQueryRequest fnaQueryRequest) {
//        long current = fnaQueryRequest.getPageNo();
//        long size = fnaQueryRequest.getPageSize();
//        // 查询数据库
//        Page<Customer> fnaPage = policyFollowService.page(new Page<>(current, size),
//                policyFollowService.getQueryWrapper(fnaQueryRequest));
//        return Result.success(fnaPage);
//    }

    /**
     * 分页获取保单发佣列表（VO）
     *
     * @param fortuneQueryRequest
     * @param request
     * @return
     */
    @PostMapping("/list/page/vo")
    @Operation(summary = "分页获取保单发佣列表")
    public Result<Page<FortuneVO>> listFortuneByPage(@RequestBody FortuneQueryRequest fortuneQueryRequest,
                                                          HttpServletRequest request) {
        long current = fortuneQueryRequest.getPageNo();
        long size = fortuneQueryRequest.getPageSize();

        // 查询数据库
        Page<Fortune> fortunePage = fortuneService.page(new Page<>(current, size),
                fortuneService.getQueryWrapper(fortuneQueryRequest));
        // 获取封装类
        return Result.success(fortuneService.getFortuneVOPage(fortunePage));
    }

}
