package com.yd.csf.api.service.impl;

import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.api.service.ApiAppointmentCheckService;
import com.yd.csf.api.service.ApiInsurantService;
import com.yd.csf.feign.dto.appointment.ApiInsurantInfoDto;
import com.yd.csf.service.model.Insurant;
import com.yd.csf.service.model.Policyholder;
import com.yd.csf.service.service.IInsurantService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import java.util.Objects;

/**
 * 受保人信息业务实现类
 */
@Slf4j
@Service
public class ApiInsurantServiceImpl implements ApiInsurantService {

    @Autowired
    private IInsurantService iInsurantService;

    @Autowired
    private ApiAppointmentCheckService apiAppointmentCheckService;

    /**
     * 受保人信息详情
     * @param appointmentBizId 预约信息主表唯一业务ID
     * @param insurantBizId 受保人信息表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiInsurantInfoDto> detail(String appointmentBizId, String insurantBizId) {
        Insurant insurant = iInsurantService.queryOne(appointmentBizId,insurantBizId);
        ApiInsurantInfoDto dto = null;
        if (!Objects.isNull(insurant)) {
            dto = new ApiInsurantInfoDto();
            BeanUtils.copyProperties(insurant,dto);
        }
        return Result.success(dto);
    }

    /**
     * 编辑受保人信息
     * @param apiInsurantInfoDto
     * @return
     */
    @Override
    public Result edit(ApiInsurantInfoDto apiInsurantInfoDto) {
        //编辑预约入参字段校验 - 受保人信息字段校验
        apiAppointmentCheckService.checkEditApiInsurantInfoDto(apiInsurantInfoDto);

        //编辑受保人信息表数据
        editInsurantData(apiInsurantInfoDto,apiInsurantInfoDto.getAppointmentBizId());
        return Result.success();
    }

    /**
     * 添加受保人信息表数据
     * @param dto
     * @param appointmentBizId
     * @return
     */
    @Override
    public Result<Insurant> addInsurantData(ApiInsurantInfoDto dto, String appointmentBizId) {
        if (Objects.isNull(dto)){
            //为空设置，方便新建暂存公用方法
            dto = new ApiInsurantInfoDto();
        }
        //受保人信息信息
        Insurant insurant = new Insurant();
        BeanUtils.copyProperties(dto,insurant);
        //生成受保人信息表唯一业务ID
        insurant.setInsurantBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_INSURANT.getCode()));
        //预约信息主表唯一业务ID
        insurant.setAppointmentBizId(appointmentBizId);
        iInsurantService.saveOrUpdate(insurant);
        return Result.success(insurant);
    }

    /**
     * 编辑受保人信息表数据
     * @param dto
     * @param appointmentBizId
     * @return
     */
    @Override
    public Result<Insurant> editInsurantData(ApiInsurantInfoDto dto, String appointmentBizId) {
        if (Objects.isNull(dto)){
            //受保人信息对象不能为空
            throw new BusinessException("受保人信息不能为空");
        }

        Result<Insurant> result = checkInsurantIsExist(dto.getInsurantBizId());
        //受保人信息信息
        Insurant insurant = result.getData();
        BeanUtils.copyProperties(dto,insurant);
        //预约信息主表唯一业务ID
        insurant.setAppointmentBizId(appointmentBizId);
        iInsurantService.saveOrUpdate(insurant);
        return Result.success(insurant);
    }

    /**
     * 校验受保人信息是否存在
     * @param insurantBizId
     * @return
     */
    public Result<Insurant> checkInsurantIsExist(String insurantBizId) {
        Insurant insurant = iInsurantService.queryOne(insurantBizId);
        if (Objects.isNull(insurant)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(insurant);
    }

}
