package com.yd.csf.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.api.service.ApiAppointmentFileService;
import com.yd.csf.feign.dto.appointmentfile.ApiAppointmentFileDto;
import com.yd.csf.feign.request.appointmentfile.ApiAppointmentFileAddRequest;
import com.yd.csf.feign.request.appointmentfile.ApiAppointmentFileEditRequest;
import com.yd.csf.feign.request.appointmentfile.ApiAppointmentFilePageRequest;
import com.yd.csf.feign.response.appointmentfile.ApiAppointmentFilePageResponse;
import com.yd.csf.service.model.AppointmentFile;
import com.yd.csf.service.model.Policyholder;
import com.yd.csf.service.service.IAppointmentFileService;
import com.yd.user.feign.response.sysuser.ApiSysUserPageResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 预约附件信息业务实现类
 */
@Slf4j
@Service
public class ApiAppointmentFileServiceImpl implements ApiAppointmentFileService {

    @Autowired
    private IAppointmentFileService iAppointmentFileService;

    /**
     * 预约附件信息分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiAppointmentFilePageResponse>> page(ApiAppointmentFilePageRequest request) {
        Page<ApiAppointmentFilePageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiAppointmentFilePageResponse> iPage = iAppointmentFileService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 根据预约业务id获取附件列表信息
     * @param appointmentBizId
     * @return
     */
    @Override
    public Result<List<ApiAppointmentFileDto>> list(String appointmentBizId) {
        List<ApiAppointmentFileDto> dtoList = new ArrayList<>();
        List<AppointmentFile> list = iAppointmentFileService.queryList(appointmentBizId);
        if (!CollectionUtils.isEmpty(list)) {
            dtoList = list.stream().map(dto -> {
                ApiAppointmentFileDto fileDto = new ApiAppointmentFileDto();
                BeanUtils.copyProperties(dto,fileDto);
                return fileDto;
            }).collect(Collectors.toList());
        }
        return Result.success(dtoList);
    }

    /**
     * 新增预约附件信息
     * @param request
     * @return
     */
    @Override
    public Result add(ApiAppointmentFileAddRequest request) {
        AppointmentFile appointmentFile = new AppointmentFile();
        BeanUtils.copyProperties(request,appointmentFile);
        appointmentFile.setAppointmentFileBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_APPOINTMENT_FILE.getCode()));
        iAppointmentFileService.saveOrUpdate(appointmentFile);
        return Result.success();
    }

    /**
     * 编辑预约附件信息
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiAppointmentFileEditRequest request) {
        Result<AppointmentFile> result = checkAppointmentFileIsExist(request.getAppointmentFileBizId());
        AppointmentFile appointmentFile = result.getData();
        appointmentFile.setFileName(request.getFileName());
        iAppointmentFileService.saveOrUpdate(appointmentFile);
        return Result.success();
    }

    /**
     * 删除预约附件
     * @param appointmentFileBizId
     * @return
     */
    @Override
    public Result del(String appointmentFileBizId) {
        Result<AppointmentFile> result = checkAppointmentFileIsExist(appointmentFileBizId);
        AppointmentFile appointmentFile = result.getData();
        iAppointmentFileService.removeById(appointmentFile.getId());
        return Result.success();
    }

    /**
     * 下载预约附件
     * @param appointmentFileBizId
     * @return
     */
    @Override
    public Result download(String appointmentFileBizId) {
        return null;
    }

    /**
     * 校验预约附件信息是否存在
     * @param appointmentFileBizId
     * @return
     */
    public Result<AppointmentFile> checkAppointmentFileIsExist(String appointmentFileBizId) {
        AppointmentFile appointmentFile = iAppointmentFileService.queryOne(appointmentFileBizId);
        if (Objects.isNull(appointmentFile)) {
            //数据不存在
            throw new BusinessException("预约附件信息不存在");
        }
        return Result.success(appointmentFile);
    }
}
