package com.yd.csf.api.controller;

import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;

import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.api.service.ApiAppointmentService;
import com.yd.csf.feign.dto.appointment.ApiAppointmentInfoDto;
import com.yd.csf.feign.response.appointment.ApiAppointmentDetailResponse;
import com.yd.csf.service.common.ErrorCode;
import com.yd.csf.service.dto.DeleteFnaRequest;
import com.yd.csf.service.dto.FnaAddRequest;
import com.yd.csf.service.dto.FnaQueryRequest;
import com.yd.csf.service.dto.FnaUpdateRequest;
import com.yd.csf.service.model.Appointment;
import com.yd.csf.service.model.Customer;
import com.yd.csf.service.model.Fna;
import com.yd.csf.service.service.CustomerService;
import com.yd.csf.service.service.FnaService;
import com.yd.csf.service.vo.FnaVO;
import com.yd.user.feign.client.sysuser.ApiSysUserFeignClient;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

/**
 * Fna接口
 *
 * @author jianan
 * @since 2025-08-31
 */
@RestController
@RequestMapping("/Fna")
@Tag(name = "流程接口")
public class ApiFnaController {

    @Resource
    private FnaService fnaService;
    @Resource
    private CustomerService customerService;
    @Resource
    private ApiAppointmentService appointmentService;
    @Resource
    private ApiSysUserFeignClient apiSysUserFeignClient;

    /**
     * 新建流程
     *
     * @param fnaAddRequest
     * @param request
     * @return
     */
    @PostMapping("/add")
    @Operation(summary = "新建流程")
    @Transactional(rollbackFor = Exception.class)
    public Result<String> addFna(@RequestBody FnaAddRequest fnaAddRequest, HttpServletRequest request) {

        if (fnaAddRequest == null) {
            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }

        Fna fna = new Fna();
        BeanUtils.copyProperties(fnaAddRequest, fna);

        // 获取Security上下文当前用户的登录信息
//        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
//        String userBizId = authUserDto.getUserBizId();

        String userBizId = "user_dMnkKPIwemvY0zhk";

        fna.setUserBizId(userBizId);
        fna.setFnaBizId(RandomStringGenerator.generateBizId16("fna"));

        fna.setCreatorId(userBizId);
        fna.setUpdaterId(userBizId);
        // 写入数据库
        boolean result = fnaService.save(fna);
        if (!result) {
            return Result.fail(ErrorCode.OPERATION_ERROR.getCode(), ErrorCode.OPERATION_ERROR.getMessage());
        }
        // 更新 fna_no
        Fna updateFna = new Fna();
        updateFna.setId(fna.getId());
        updateFna.setFnaNo(fna.getId());
        fnaService.updateById(updateFna);

        // 返回新写入的数据 id
        String newFnaId = fna.getFnaBizId();
        return Result.success(newFnaId);
    }

    /**
     * 删除流程
     *
     * @param deleteRequest
     * @param request
     * @return
     */
    @PostMapping("/delete")
    @Operation(summary = "删除流程")
    public Result<Boolean> deleteFna(@RequestBody DeleteFnaRequest deleteRequest, HttpServletRequest request) {
        if (deleteRequest == null || deleteRequest.getId() == null) {
            throw new BusinessException(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }

        Long id = deleteRequest.getId();
        // 判断是否存在
        Fna oldFna = fnaService.getById(id);
        if (oldFna == null) {
            throw new BusinessException(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
        }
        // 仅本人或管理员可删除
//        if (!oldFna.getUserId().equals(user.getId()) && !userService.isAdmin(request)) {
//            throw new BusinessException(ErrorCode.NO_AUTH_ERROR);
//        }
        // 操作数据库
        Fna updateFna = new Fna();
        updateFna.setId(id);
        updateFna.setIsDeleted(1);
        boolean result = fnaService.updateById(updateFna);
        if (!result) {
            return Result.fail(ErrorCode.OPERATION_ERROR.getCode(), ErrorCode.OPERATION_ERROR.getMessage());
        }
        return Result.success(true);
    }

    /**
     * 更新流程
     *
     * @param fnaUpdateRequest
     * @return
     */
    @PostMapping("/update")
    @Operation(summary = "更新流程")
    public Result<Boolean> updateFna(@RequestBody FnaUpdateRequest fnaUpdateRequest) {
        if (fnaUpdateRequest == null || StringUtils.isBlank(fnaUpdateRequest.getFnaBizId())) {
            throw new BusinessException(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }

        // 判断是否存在
        String fnaBizId = fnaUpdateRequest.getFnaBizId();
        Fna fna = fnaService.getByBizId(fnaBizId);
        if (fna == null) {
            throw new BusinessException(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
        }

        BeanUtils.copyProperties(fnaUpdateRequest, fna,"fnaBizId");

        if (StringUtils.isNotBlank(fnaUpdateRequest.getCustomerBizId())) {
            Customer customer = customerService.getByCustomerBizId(fnaUpdateRequest.getCustomerBizId());
            fna.setCustomerName(customer.getName());
        }
        if (StringUtils.isNotBlank(fnaUpdateRequest.getAppointmentBizId())) {
            Result<ApiAppointmentInfoDto> apiAppointmentInfoDtoResult = appointmentService.mainDetail(fnaUpdateRequest.getAppointmentBizId());
            if (apiAppointmentInfoDtoResult.getData() != null) {
                ApiAppointmentInfoDto apiAppointmentInfoDto = apiAppointmentInfoDtoResult.getData();
                fna.setAppointmentNo(apiAppointmentInfoDto.getAppointmentNo());
            }
        }

        // 操作数据库
        boolean result = fnaService.updateById(fna);
        if (!result) {
            return Result.fail(ErrorCode.OPERATION_ERROR.getCode(), ErrorCode.OPERATION_ERROR.getMessage());
        }
        return Result.success(true);
    }

    /**
     * 根据 id 获取fna（封装类）
     *
     * @param id
     * @return
     */
    @GetMapping("/get/vo")
    @Operation(summary = "流程详情")
    public Result<FnaVO> getFnaVOById(long id, HttpServletRequest request) {
        if (id == 0) {
            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }
        // 查询数据库
        Fna fna = fnaService.getById(id);
        if (fna == null) {
            return Result.fail(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
        }

        // 获取封装类
        return Result.success(fnaService.getFnaVO(fna));
    }

    /**
     * 分页获取fna列表（仅管理员可用）
     *
     * @param fnaQueryRequest
     * @return
     */
    @PostMapping("/list/page")
    @Operation(summary = "分页获取流程列表")
    public Result<Page<FnaVO>> listFnaByPage(@RequestBody FnaQueryRequest fnaQueryRequest) {
        long current = fnaQueryRequest.getPageNo();
        long size = fnaQueryRequest.getPageSize();
        // 查询数据库
        Page<Fna> fnaPage = fnaService.page(new Page<>(current, size),
                fnaService.getQueryWrapper(fnaQueryRequest));
        return Result.success(fnaService.getFnaVOPage(fnaPage));
    }

    /**
     * 分页获取fna列表（封装类）
     *
     * @param fnaQueryRequest
     * @param request
     * @return
     */
//    @PostMapping("/list/page/vo")
//    public Result<Page<Fna>> listFnaVOByPage(@RequestBody FnaQueryRequest fnaQueryRequest,
//                                             HttpServletRequest request) {
//        long current = fnaQueryRequest.getPageNo();
//        long size = fnaQueryRequest.getPageSize();
//
//        // 查询数据库
//        Page<Fna> fnaPage = fnaService.page(new Page<>(current, size),
//                fnaService.getQueryWrapper(fnaQueryRequest));
//        // 获取封装类
////        return Result.success(fnaService.getFnaVOPage(fnaPage, request));
//
//        return Result.success(fnaPage);
//    }


}
