package com.yd.csf.service.service;

import com.yd.csf.service.enums.CommissionStatusEnum;
import com.yd.csf.service.model.Commission;
import com.yd.csf.service.model.CommissionExpected;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.List;
import java.util.stream.Collectors;

@Service
@Slf4j
public class CommissionAsyncService {
    @Resource
    private CommissionExpectedService commissionExpectedService;
    @Resource
    private CommissionService commissionService;

    @Async("commonAsyncExecutor")
    @Transactional(rollbackFor = Exception.class)
    public void commissionCompareBatch(List<Commission> entities) {
        // 比对逻辑...
        log.info("开始异步比对，数据量：{}", entities.size());

        List<String> policyNoList = entities.stream()
                .map(Commission::getPolicyNo)
                .collect(Collectors.toList());

        List<CommissionExpected> expectedList = commissionExpectedService.lambdaQuery()
                .in(CommissionExpected::getPolicyNo, policyNoList)
                .list();

        // 执行比对...
    }

    public void commissionCompare(Commission commission) {
        log.info("开始执行比对事务，来佣ID: {}, 保单号: {}",
                commission.getId(), commission.getPolicyNo());

        CommissionExpected commissionExpected;
        try {
            commissionExpected = commissionService.queryByCommission(commission);
            // 对比预计来佣
            commissionService.compareWithExpected(commission.getPremium(), commission, commissionExpected);
            commissionExpectedService.updateById(commissionExpected);

            // 比对成功，更新状态
            commission.setStatus(CommissionStatusEnum.COMPARE_SUCCESS.getItemValue());

        } catch (Exception e) {
            // 预计来佣不存在或比对失败
            commission.setStatus(CommissionStatusEnum.COMPARE_FAIL.getItemValue());
            commission.setRemark(e.getMessage());
            log.warn("来佣比对失败，保单号: {}，原因: {}",
                    commission.getPolicyNo(), e.getMessage());
        }

        // 更新来佣的比对状态（在比对事务中）
        commissionService.updateById(commission);
    }
}
