package com.yd.csf.service.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.service.common.ErrorCode;
import com.yd.csf.service.dto.PolicyFollowAddRequest;
import com.yd.csf.service.dto.PolicyFollowQueryRequest;
import com.yd.csf.service.dto.PolicyFollowUpdateRequest;
import com.yd.csf.service.model.Policy;
import com.yd.csf.service.model.PolicyBroker;
import com.yd.csf.service.model.PolicyFollow;
import com.yd.csf.service.service.PolicyBrokerService;
import com.yd.csf.service.service.PolicyFollowService;
import com.yd.csf.service.dao.PolicyFollowMapper;
import com.yd.csf.service.service.PolicyService;
import com.yd.csf.service.vo.PolicyFollowVO;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
* @author Zhang Jianan
* @description 针对表【policy_follow(新单跟进表)】的数据库操作Service实现
* @createDate 2025-09-16 18:18:06
*/
@Service
public class PolicyFollowServiceImpl extends ServiceImpl<PolicyFollowMapper, PolicyFollow>
    implements PolicyFollowService{

    @Resource
    private PolicyBrokerService policyBrokerService;
    @Resource
    private PolicyService policyService;

    @Override
    public PolicyFollow getByPolicyBizId(String policyBizId) {
        return this.getOne(new QueryWrapper<PolicyFollow>().eq("policy_biz_id", policyBizId));
    }

    @Override
    public QueryWrapper<PolicyFollow> getQueryWrapper(PolicyFollowQueryRequest policyFollowQueryRequest) {
        QueryWrapper<PolicyFollow> queryWrapper = new QueryWrapper<>();
        if (policyFollowQueryRequest == null) {
            return queryWrapper;
        }
        String policyNo = policyFollowQueryRequest.getPolicyNo();
        String customerName = policyFollowQueryRequest.getCustomerName();
        String customerBizId = policyFollowQueryRequest.getCustomerBizId();
        String status = policyFollowQueryRequest.getStatus();
        String insurer = policyFollowQueryRequest.getInsurer();
        String productName = policyFollowQueryRequest.getProductName();
        Date signDateStart = policyFollowQueryRequest.getSignDateStart();
        Date signDateEnd = policyFollowQueryRequest.getSignDateEnd();

        queryWrapper.eq(ObjectUtils.isNotEmpty(policyNo), "policy_no", policyNo);
        queryWrapper.eq(ObjectUtils.isNotEmpty(customerName), "customer_name", customerName);
        queryWrapper.eq(ObjectUtils.isNotEmpty(customerBizId), "customer_biz_id", customerBizId);
        queryWrapper.eq(ObjectUtils.isNotEmpty(status), "status", status);
        queryWrapper.eq(ObjectUtils.isNotEmpty(insurer), "insurance", insurer);
        queryWrapper.eq(ObjectUtils.isNotEmpty(productName), "product_code", productName);
        queryWrapper.ge(ObjectUtils.isNotEmpty(signDateStart), "sign_date", signDateStart);
        queryWrapper.le(ObjectUtils.isNotEmpty(signDateEnd), "sign_date", signDateEnd);

        // 排序
        queryWrapper.orderByDesc("id");

        return queryWrapper;
    }

    @Override
    public Page<PolicyFollowVO> getPolicyFollowVOPage(Page<PolicyFollow> policyFollowPage) {
        List<PolicyFollow> policyFollowList = policyFollowPage.getRecords();
        Page<PolicyFollowVO> policyFollowVOPage = new Page<>(policyFollowPage.getCurrent(), policyFollowPage.getSize(), policyFollowPage.getTotal());
        if (CollUtil.isEmpty(policyFollowList)) {
            return policyFollowVOPage;
        }
        // 对象列表 => 封装对象列表
        List<PolicyFollowVO> policyFollowVOList = policyFollowList.stream().map(PolicyFollowVO::objToVo).collect(Collectors.toList());

        // 关联查询转介人
        List<String> policyBizIdList = policyFollowVOList.stream().map(PolicyFollowVO::getPolicyBizId).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(policyBizIdList)) {
            // 转介人业务id列表 => 转介人列表
            List<PolicyBroker> policyBrokerList = policyBrokerService.list(new LambdaQueryWrapper<PolicyBroker>().in(PolicyBroker::getPolicyBizId, policyBizIdList));
            // 转介人列表 => 转介人业务id列表
            Map<String, List<PolicyBroker>> brokerMap = policyBrokerList.stream().collect(Collectors.groupingBy(PolicyBroker::getPolicyBizId));
            // 转介人列表 => 封装转介人列表
            policyFollowVOList.forEach(policyFollowVO -> {
                List<PolicyBroker> brokerList = brokerMap.get(policyFollowVO.getPolicyBizId());
                policyFollowVO.setBrokerList(brokerList);
            });
        }

        return policyFollowVOPage.setRecords(policyFollowVOList);
    }

    @Override
    public Boolean updatePolicyFollow(PolicyFollowUpdateRequest policyFollowUpdateRequest) {
        if (policyFollowUpdateRequest == null || ObjectUtils.isEmpty(policyFollowUpdateRequest.getPolicyBizId())) {
            throw new BusinessException(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }
        String policyBizId = policyFollowUpdateRequest.getPolicyBizId();
        PolicyFollow policyFollow = getByPolicyBizId(policyBizId);
        if (policyFollow == null) {
            throw new BusinessException(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
        }
        BeanUtils.copyProperties(policyFollowUpdateRequest, policyFollow, "policyBizId");
        return updateById(policyFollow);
    }

    @Override
    public PolicyFollowVO getPolicyFollowVO(PolicyFollow policyFollow) {
        if (policyFollow == null) {
            return null;
        }
        PolicyFollowVO policyFollowVO = PolicyFollowVO.objToVo(policyFollow);
        // 关联查询转介人
        QueryWrapper<PolicyBroker> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("policy_biz_id", policyFollow.getPolicyBizId());
        List<PolicyBroker> policyBrokerList = policyBrokerService.list(queryWrapper);
        policyFollowVO.setBrokerList(policyBrokerList);

        return policyFollowVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Map<String, Object> addPolicyFollow(PolicyFollowAddRequest policyFollowAddRequest) {
        PolicyFollow policyFollow = new PolicyFollow();
        BeanUtils.copyProperties(policyFollowAddRequest, policyFollow);
        // 生成新单编号
        String policyBizId = RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_POLICY.getCode());
        policyFollow.setPolicyBizId(policyBizId);
        // 创建 Policy对象
        Policy policy = new Policy();
        BeanUtils.copyProperties(policyFollowAddRequest, policy);
        policy.setPolicyBizId(policyBizId);
        // 保存转介人
        List<PolicyBroker> policyBrokerList = policyFollowAddRequest.getBrokerList();
        if (CollUtil.isNotEmpty(policyBrokerList)) {
            policyBrokerList.forEach(policyBroker -> {
                policyBroker.setPolicyBizId(policyBizId);
            });
        }

        this.save(policyFollow);
        policyService.save(policy);
        policyBrokerService.saveBatch(policyBrokerList);

        // 返回新单跟进唯一业务ID
        return Collections.singletonMap("policyBizId", policyBizId);
    }


}




