package com.yd.csf.service.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.csf.service.dto.FortuneDownloadRequest;
import com.yd.csf.service.dto.FortuneExportDTO;
import com.yd.csf.service.dto.FortuneQueryRequest;
import com.yd.csf.service.dto.FortuneUpdateRequest;
import com.yd.csf.service.model.Fortune;
import com.yd.csf.service.service.FortuneService;
import com.yd.csf.service.dao.FortuneMapper;
import com.yd.csf.service.vo.FortuneVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author Zhang Jianan
 * @description 针对表【fortune(保单发佣表)】的数据库操作Service实现
 * @createDate 2025-09-22 13:22:08
 */
@Service
public class FortuneServiceImpl extends ServiceImpl<FortuneMapper, Fortune>
        implements FortuneService {

    @Override
    public Wrapper<Fortune> getQueryWrapper(FortuneQueryRequest fortuneQueryRequest) {
        QueryWrapper<Fortune> queryWrapper = new QueryWrapper<>();
        if (fortuneQueryRequest == null) {
            return queryWrapper;
        }
        String status = fortuneQueryRequest.getStatus();
        String policyNo = fortuneQueryRequest.getPolicyNo();
        queryWrapper.eq(StringUtils.isNotBlank(status), "status", status);
        queryWrapper.eq(StringUtils.isNotBlank(policyNo), "policy_no", policyNo);

        queryWrapper.orderByAsc("id");
        return queryWrapper;
    }

    @Override
    public Page<FortuneVO> getFortuneVOPage(Page<Fortune> fortunePage) {
        List<Fortune> fortuneList = fortunePage.getRecords();
        Page<FortuneVO> fortuneVOPage = new Page<>(fortunePage.getCurrent(), fortunePage.getSize());
        if (CollUtil.isEmpty(fortuneList)) {
            return fortuneVOPage;
        }

        List<FortuneVO> fortuneVOList = fortuneList.stream().map(fortune -> {
            FortuneVO fortuneVO = new FortuneVO();
            BeanUtils.copyProperties(fortune, fortuneVO);
            return fortuneVO;
        }).collect(Collectors.toList());

        fortuneVOPage.setRecords(fortuneVOList);
        return fortuneVOPage;
    }

    @Override
    public Boolean updateFortune(FortuneUpdateRequest fortuneUpdateRequest) {
        String fortuneBizId = fortuneUpdateRequest.getFortuneBizId();
        Fortune fortune = this.getOne(new QueryWrapper<Fortune>().eq("fortune_biz_id", fortuneBizId));
        if (fortune == null) {
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(), "发佣数据不存在");
        }
        BeanUtils.copyProperties(fortuneUpdateRequest, fortune);

        // 获取当前登录用户
        AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
        String loginUserId = currentLoginUser.getId().toString();

        fortune.setUpdaterId(loginUserId);
        fortune.setUpdateTime(new Date());

        return this.updateById(fortune);
    }

    @Override
    public void downloadFortune(FortuneDownloadRequest fortuneDownloadRequest, HttpServletResponse response) throws IOException {
        List<String> fortuneBizIdList = fortuneDownloadRequest.getFortuneBizIdList();
        if (CollUtil.isNotEmpty(fortuneBizIdList)) {
            List<Fortune> fortuneList = this.list(new QueryWrapper<Fortune>().in("fortune_biz_id", fortuneBizIdList));
            if (CollUtil.isNotEmpty(fortuneList)) {
                // 转换为导出DTO
                List<FortuneExportDTO> exportDTOList = fortuneList.stream().map(fortune -> {
                    FortuneExportDTO exportDTO = new FortuneExportDTO();
                    BeanUtils.copyProperties(fortune, exportDTO);
                    return exportDTO;
                }).collect(Collectors.toList());

                // 设置响应头
                response.setContentType("application/vnd.ms-excel;charset=UTF-8");
                response.setCharacterEncoding("utf-8");
                String encode = URLEncoder.encode("保单发佣.xlsx", "UTF-8");
                response.setHeader("Content-Disposition", "attachment;filename=" + "fortune_"+System.currentTimeMillis() + ".xlsx");

                // 导出
                try {
                    EasyExcel.write(response.getOutputStream(), FortuneExportDTO.class).sheet("保单发佣").doWrite(exportDTOList);
                } catch (IOException e) {
                    log.error("导出保单发佣数据失败", e);
                    throw new BusinessException(ResultCode.FAIL.getCode(), "导出失败");
                }
            }
        }
    }
}




