package com.yd.email.api.service.impl;

import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.DateUtil;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.email.api.service.ApiEmailSendService;
import com.yd.email.api.service.XxlJobService;
import com.yd.email.feign.enums.EmailTaskStatusEnum;
import com.yd.email.feign.request.ApiSendEmailRequest;
import com.yd.email.feign.response.ApiSendEmailResponse;
import com.yd.email.service.model.EmailTask;
import com.yd.email.service.model.EmailTaskRecipients;
import com.yd.email.service.service.IEmailTaskService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 邮件发送实现类
 */
@Service
@Slf4j
public class ApiEmailSendServiceImpl implements ApiEmailSendService {

    @Autowired
    private XxlJobService xxlJobService;

    @Autowired
    private IEmailTaskService iEmailTaskService;

    /**
     * 发送邮件
     * @param request
     * @return
     */
    @Override
    public Result<ApiSendEmailResponse> sendEmail(ApiSendEmailRequest request) {
        ApiSendEmailResponse response = new ApiSendEmailResponse();
        try {
            //保存邮件任务到数据库
            EmailTask mailTask = new EmailTask();
            //邮件任务唯一业务ID
            mailTask.setTaskBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EMAIL_TASK.getCode()));
            //任务名称：邮件主题 + 邮件发送任务
            mailTask.setTaskName(request.getSubject() + "邮件发送任务");
            //关联发件人唯一业务ID
            mailTask.setSenderBizId(request.getSenderBizId());
            //发件人邮箱
            mailTask.setSendEmail(request.getSendEmail());
            //邮件主题
            mailTask.setSubject(request.getSubject());
            //邮件内容
            mailTask.setContent(request.getContent());
            //发送邮件的附件路径（多个用分号分隔）
            mailTask.setAttachmentPath(request.getAttachmentPath());
            //计划发送时间（为空表示立即发送，不为空表示定时发送）
            mailTask.setScheduleTime(request.getScheduleTime());
            //任务状态：计划发送时间（为空表示立即发送状态为发送中，不为空表示发送状态为定时发送）
            String taskStatus = !Objects.isNull(request.getScheduleTime()) ? EmailTaskStatusEnum.SCHEDULED.getItemValue() : EmailTaskStatusEnum.SENDING.getItemValue();
            mailTask.setStatus(taskStatus);
            iEmailTaskService.saveOrUpdate(mailTask);

            //邮件任务唯一业务ID
            String taskBizId = mailTask.getTaskBizId();

            // 保存收件人信息
            if (!CollectionUtils.isEmpty(request.getRecipientsDtoList())) {
                List<EmailTaskRecipients> recipientsList = request.getRecipientsDtoList()
                        .stream().map(dto -> {
                            EmailTaskRecipients mailRecipient = new EmailTaskRecipients();
                            //邮件任务唯一业务ID
                            mailRecipient.setTaskBizId(taskBizId);
                            mailRecipient.setStatus(taskStatus);
                            mailRecipient.setContactBizId(dto.getContactBizId());
                            mailRecipient.setReceiveEmail(dto.getReceiveEmail());
                            //抄送人邮箱（多个用分号分隔）
                            mailRecipient.setCcEmail(!CollectionUtils.isEmpty(dto.getCcEmailList()) ? String.join(";",dto.getCcEmailList()) : "");
                            return mailRecipient;
                        }).collect(Collectors.toList());
            }

            //计划发送时间（为空表示立即发送，不为空表示定时发送）
            String jobId = "";
            if (!Objects.isNull(request.getScheduleTime())) {
                //创建XXL-Job定时任务
                jobId = xxlJobService.addScheduleJob(taskBizId, DateUtil.convertDateByLocalDateTime(request.getScheduleTime()));
            }
            response.setJobId(jobId);
            response.setScheduleTime(request.getScheduleTime());
            response.setTaskBizId(taskBizId);
            return Result.success(response);
        } catch (Exception e) {
            log.error("创建邮件发送任务失败", e);
            throw new BusinessException("创建邮件发送任务失败");
        }
    }
}
