package com.yd.email.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.email.api.service.ApiEmailVariableGroupService;
import com.yd.email.api.service.ApiRelVariableGroupService;
import com.yd.email.feign.dto.ApiEmailVariableDto;
import com.yd.email.feign.request.ApiEmailVariableGroupAddRequest;
import com.yd.email.feign.request.ApiEmailVariableGroupEditRequest;
import com.yd.email.feign.request.ApiEmailVariableGroupPageRequest;
import com.yd.email.feign.response.ApiEmailVariableGroupDetailResponse;
import com.yd.email.feign.response.ApiEmailVariableGroupPageResponse;
import com.yd.email.service.dto.EmailVariableGroupDto;
import com.yd.email.service.model.EmailVariableGroup;
import com.yd.email.service.service.IEmailVariableGroupService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;

/**
 * 变量分组信息实现类
 */
@Slf4j
@Service
public class ApiEmailVariableGroupServiceImpl implements ApiEmailVariableGroupService {

    @Autowired
    private IEmailVariableGroupService iEmailVariableGroupService;

    @Autowired
    private ApiRelVariableGroupService apiRelVariableGroupService;

    /**
     * 分页查询-变量分组
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiEmailVariableGroupPageResponse>> page(ApiEmailVariableGroupPageRequest request) {
        Page<ApiEmailVariableGroupPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiEmailVariableGroupPageResponse> iPage = iEmailVariableGroupService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 添加-变量分组
     * @param request
     * @return
     */
    @Override
    public Result add(ApiEmailVariableGroupAddRequest request) {
        //校验变量分组名称的唯一性
        List<EmailVariableGroup> list = iEmailVariableGroupService.queryList(EmailVariableGroupDto.builder()
                .groupName(request.getGroupName())
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("变量分组名称已存在！");
        }
        EmailVariableGroup emailVariableGroup = new EmailVariableGroup();
        BeanUtils.copyProperties(request,emailVariableGroup);
        //设置变量分组唯一业务ID
        emailVariableGroup.setVariableGroupBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EMAIL_VARIABLE_GROUP.getCode()));
        iEmailVariableGroupService.saveOrUpdate(emailVariableGroup);

        //保存绑定的变量集合关系
        apiRelVariableGroupService.saveRelVariableGroup(request.getVariableBizIdList(),
                emailVariableGroup.getVariableGroupBizId());
        return Result.success();
    }

    /**
     * 编辑-变量分组
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiEmailVariableGroupEditRequest request) {
        //校验变量分组是否存在
        Result<EmailVariableGroup> result = checkEmailVariableGroupIsExist(request.getVariableGroupBizId());
        EmailVariableGroup emailVariableGroup = result.getData();
        //校验变量分组名称的唯一性
        List<EmailVariableGroup> list = iEmailVariableGroupService.queryList(EmailVariableGroupDto.builder()
                .groupName(request.getGroupName())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("变量分组名称已存在！");
        }
        BeanUtils.copyProperties(request,emailVariableGroup);
        iEmailVariableGroupService.saveOrUpdate(emailVariableGroup);

        //保存绑定的变量集合关系
        apiRelVariableGroupService.saveRelVariableGroup(request.getVariableBizIdList(),
                emailVariableGroup.getVariableGroupBizId());
        return Result.success();
    }

    /**
     * 详情-变量分组
     * @param variableGroupBizId 变量分组唯一业务ID
     * @return
     */
    @Override
    public Result<ApiEmailVariableGroupDetailResponse> detail(String variableGroupBizId) {
        //校验变量分组是否存在
        Result<EmailVariableGroup> result = checkEmailVariableGroupIsExist(variableGroupBizId);
        EmailVariableGroup emailVariableGroup = result.getData();
        ApiEmailVariableGroupDetailResponse response = new ApiEmailVariableGroupDetailResponse();
        BeanUtils.copyProperties(emailVariableGroup,response);
        response.setEmailVariableDtoList(apiRelVariableGroupService.getEmailVariableDtoList(variableGroupBizId));
        return Result.success(response);
    }

    /**
     * 删除-变量分组
     * @param variableGroupBizId 变量分组唯一业务ID
     * @return
     */
    @Override
    public Result del(String variableGroupBizId) {
        //校验变量分组是否存在
        Result<EmailVariableGroup> result = checkEmailVariableGroupIsExist(variableGroupBizId);
        EmailVariableGroup emailVariableGroup = result.getData();
        //删变量分组
        iEmailVariableGroupService.removeById(emailVariableGroup.getId());

        //删变量分组关系
        apiRelVariableGroupService.delRelVariableGroup(variableGroupBizId);

        return Result.success();
    }

    /**
     * 校验变量分组是否存在
     * @param variableGroupBizId
     * @return
     */
    public Result<EmailVariableGroup> checkEmailVariableGroupIsExist(String variableGroupBizId) {
        EmailVariableGroup emailVariableGroup = iEmailVariableGroupService.queryOne(variableGroupBizId);
        if (Objects.isNull(emailVariableGroup)) {
            //数据不存在
            throw new BusinessException("变量分组信息不存在！");
        }
        return Result.success(emailVariableGroup);
    }
}
