package com.yd.email.api.service.impl;

import com.yd.common.result.Result;
import com.yd.email.api.service.ApiRelVariableGroupService;
import com.yd.email.feign.dto.ApiEmailVariableDto;
import com.yd.email.service.dto.EmailVariableDto;
import com.yd.email.service.model.EmailVariable;
import com.yd.email.service.model.RelVariableGroup;
import com.yd.email.service.service.IEmailVariableService;
import com.yd.email.service.service.IRelVariableGroupService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 变量分组关系实现类
 */
@Slf4j
@Service
public class ApiRelVariableGroupServiceImpl implements ApiRelVariableGroupService {

    @Autowired
    private IRelVariableGroupService iRelVariableGroupService;

    @Autowired
    private IEmailVariableService iEmailVariableService;

    /**
     * 保存变量分组关系
     * @param variableBizIdList
     * @param variableGroupBizId
     * @return
     */
    @Override
    public Result saveRelVariableGroup(List<String> variableBizIdList, String variableGroupBizId) {
        //先删后增加
        iRelVariableGroupService.delRelByVariableGroupBizId(variableGroupBizId);

        //增加关系
        if (!CollectionUtils.isEmpty(variableBizIdList)) {
            List<RelVariableGroup> saveList = variableBizIdList.stream().map(variableBizId -> {
                RelVariableGroup group = new RelVariableGroup();
                group.setVariableBizId(variableBizId);
                group.setVariableGroupBizId(variableGroupBizId);
                return group;
            }).collect(Collectors.toList());
            iRelVariableGroupService.saveOrUpdateBatch(saveList);
        }
        return Result.success();
    }

    /**
     * 根据变量分组业务id删除关系
     * @param variableGroupBizId
     * @return
     */
    @Override
    public Result delRelVariableGroup(String variableGroupBizId) {
        iRelVariableGroupService.delRelByVariableGroupBizId(variableGroupBizId);
        return Result.success();
    }

    /**
     * 根据分组变量业务id查询绑定的变量集合列表
     * @param variableGroupBizId
     * @return
     */
    @Override
    public List<ApiEmailVariableDto> getEmailVariableDtoList(String variableGroupBizId) {
        List<ApiEmailVariableDto> dtoList = new ArrayList<>();
        //查询关系列表
        List<RelVariableGroup> relVariableGroupList = iRelVariableGroupService.queryList(variableGroupBizId);
        if (!CollectionUtils.isEmpty(relVariableGroupList)) {
            List<String> variableBizIdList = relVariableGroupList.stream()
                    .map(RelVariableGroup::getVariableBizId).collect(Collectors.toList());

            List<EmailVariable> emailVariableList = iEmailVariableService.queryList(EmailVariableDto.builder()
                    .variableBizIdList(variableBizIdList).build());

            if (!CollectionUtils.isEmpty(emailVariableList)) {
                dtoList = emailVariableList.stream().map(dto -> {
                    ApiEmailVariableDto apiEmailVariableDto = new ApiEmailVariableDto();
                    BeanUtils.copyProperties(dto,apiEmailVariableDto);
                    return apiEmailVariableDto;
                }).collect(Collectors.toList());
            }
        }
        return dtoList;
    }

}
