package com.yd.email.api.service.impl;

import com.alibaba.fastjson.JSON;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.DateUtil;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.email.api.service.*;
import com.yd.email.feign.dto.ApiEmailSenderConfigDto;
import com.yd.email.feign.dto.ApiSendMailDto;
import com.yd.email.feign.enums.EmailTaskStatusEnum;
import com.yd.email.feign.request.ApiSendEmailRequest;
import com.yd.email.feign.response.ApiSendEmailResponse;
import com.yd.email.feign.utils.StringUtil;
import com.yd.email.service.model.EmailTask;
import com.yd.email.service.model.EmailTaskRecipients;
import com.yd.email.service.service.IEmailTaskRecipientsService;
import com.yd.email.service.service.IEmailTaskService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * 邮件发送实现类
 */
@Service
@Slf4j
public class ApiEmailSendServiceImpl implements ApiEmailSendService {

    @Autowired
    private XxlJobService xxlJobService;

    @Autowired
    private IEmailTaskService iEmailTaskService;

    @Autowired
    private IEmailTaskRecipientsService iEmailTaskRecipientsService;

    @Autowired
    private ApiEmailSenderConfigService apiEmailSenderConfigService;

    @Autowired
    private ApiEmailService apiEmailService;

    @Autowired
    private ApiEmailTaskRecipientsService apiEmailTaskRecipientsService;

    /**
     * 发送邮件
     * @param request
     * @return
     */
    @Override
    public Result<ApiSendEmailResponse> sendEmail(ApiSendEmailRequest request) {
        ApiSendEmailResponse response = new ApiSendEmailResponse();
        try {
            //保存邮件任务到数据库
            EmailTask mailTask = new EmailTask();
            //邮件任务唯一业务ID
            mailTask.setTaskBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EMAIL_TASK.getCode()));
            //任务名称：邮件主题 + 邮件发送任务
            mailTask.setTaskName(request.getSubject() + "邮件发送任务");
            //关联发件人唯一业务ID
            mailTask.setSenderBizId(request.getSenderBizId());
            //发件人邮箱
            mailTask.setSendEmail(request.getSendEmail());
            //邮件主题
            mailTask.setSubject(request.getSubject());
            //邮件内容
            mailTask.setContent(request.getContent());
            //发送邮件的附件路径（多个用分号分隔）
            mailTask.setAttachmentPath(request.getAttachmentPath());
            //计划发送时间（为空表示立即发送，不为空表示定时发送）
            mailTask.setScheduleTime(request.getScheduleTime());
            //任务状态：计划发送时间（为空表示立即发送状态为发送中，不为空表示发送状态为定时发送）
            String taskStatus = !Objects.isNull(request.getScheduleTime()) ? EmailTaskStatusEnum.SCHEDULED.getItemValue() : EmailTaskStatusEnum.SENDING.getItemValue();
            mailTask.setStatus(taskStatus);
            //变量分组唯一业务ID
            mailTask.setVariableGroupBizId(request.getVariableGroupBizId());
            //邮件内容的填充的变量字段（数组分号分隔，英文字段名）
//            mailTask.setFillVariable(!CollectionUtils.isEmpty(request.getFillVariableList()) ? String.join(";") : "");
            //导入临时表的当前会话id（前端使用，用于查询。存于前端缓存）（冗余字段，选择联系人和导入联系人excel都有值）
            mailTask.setSessionId(request.getSessionId());
            iEmailTaskService.saveOrUpdate(mailTask);

            //邮件任务唯一业务ID
            String taskBizId = mailTask.getTaskBizId();
            //变量分组唯一业务ID不为空，来源是1-excel导入联系人 为空是0-选择联系人导入
            Integer source = StringUtils.isNotBlank(request.getVariableGroupBizId()) ? 1 : 0;

            // 保存收件人信息
            //根据当前当前会话id和数据来源查询导入数据，保存到邮件任务收件人关联表中，供后续使用
            Result<List<EmailTaskRecipients>> result = apiEmailTaskRecipientsService
                    .saveEmailTaskRecipientsBySessionId(taskBizId,taskStatus,request.getSessionId(),source);
            List<EmailTaskRecipients> recipientsList = result.getData();

            //计划发送时间（为空表示立即发送，不为空表示定时发送）
            String jobId = "";
            if (!Objects.isNull(request.getScheduleTime())) {
                //不为空表示定时发送：创建XXL-Job定时任务
                jobId = xxlJobService.addScheduleJob(taskBizId, DateUtil.convertDateByLocalDateTime(request.getScheduleTime()));
            }else {
                //为空表示立即发送
                sendImmediatelyEmail(mailTask,recipientsList);
            }
            response.setJobId(jobId);
            response.setScheduleTime(request.getScheduleTime());
            response.setTaskBizId(taskBizId);
            return Result.success(response);
        } catch (Exception e) {
            log.error("创建邮件发送任务失败", e);
            throw new BusinessException("发送邮件异常");
        }
    }

    /**
     * 立即发送邮件
     * @param emailTask
     * @param recipients
     * @return
     */
    public Result sendImmediatelyEmail(EmailTask emailTask,List<EmailTaskRecipients> recipients) {
        try {
            //调用邮件服务发送邮件入参DTO
            ApiSendMailDto apiSendMailDto = new ApiSendMailDto();
            BeanUtils.copyProperties(emailTask,apiSendMailDto);
            //根据发件人唯一业务ID获取发件人信息和发件人绑定的服务商信息
            Result<ApiEmailSenderConfigDto> result = apiEmailSenderConfigService.getApiEmailSenderConfigDto(emailTask.getSenderBizId());
            ApiEmailSenderConfigDto apiEmailSenderConfigDto = result.getData();
            BeanUtils.copyProperties(apiEmailSenderConfigDto,apiSendMailDto);

            //该任务下的所有收件人信息
            log.info("查询该任务下的所有收件人信息: {}: " + JSON.toJSONString(recipients));

            // 初始化成功和失败计数器
            int successCount = 0;
            int failCount = 0;

            // 遍历所有收件人，逐个发送邮件
            for (EmailTaskRecipients recipient : recipients) {
                try {
                    // 处理抄送人列表：将数据库中的逗号分隔字符串转换为List
                    List<String> ccEmailList = StringUtils.isNotBlank(recipient.getCcEmail()) ?
                            Arrays.asList(recipient.getCcEmail().split(";")) :
                            new ArrayList<>();
                    log.info("处理抄送人列表：将数据库中的逗号分隔字符串转换为List: {}: " + JSON.toJSONString(ccEmailList));
                    //收件人邮箱（单个）
                    apiSendMailDto.setReceiveEmail(recipient.getReceiveEmail());
                    //抄送人邮箱列表
                    apiSendMailDto.setCcEmailList(ccEmailList);
                    //邮件内容
                    if (StringUtil.hasPlaceholdersSimple(apiSendMailDto.getContent())) {
                        //检测内容是否有占位符，有占位符就替换具体内容,每收件人的占位符内容是不一样的。
                        apiSendMailDto.setContent(StringUtil.replacePlaceholders(apiSendMailDto.getContent(),
                                recipient.getVariables()));
                    }
                    // 调用邮件服务发送邮件
                    apiEmailService.sendMail(apiSendMailDto);

                    // 发送成功：更新收件人状态为成功
                    recipient.setStatus(EmailTaskStatusEnum.SUCCESSFUL.getItemValue());
                    recipient.setSendTime(LocalDateTime.now());  // 记录实际发送时间
                    iEmailTaskRecipientsService.saveOrUpdate(recipient);
                    successCount++;  // 成功计数加1

                } catch (Exception e) {
                    // 发送失败：记录错误日志
                    log.error("发送邮件失败: {}", recipient.getReceiveEmail(), e);
                    // 更新收件人状态为失败
                    recipient.setStatus(EmailTaskStatusEnum.FAILED.getItemValue());
                    recipient.setErrorMsg(e.getMessage());  // 保存错误信息
                    iEmailTaskRecipientsService.saveOrUpdate(recipient);
                    failCount++;  // 失败计数加1
                }
            }

            // 根据发送结果更新邮件任务状态
            emailTask.setStatus(failCount == 0 ?
                    EmailTaskStatusEnum.ALL_SUCCESSFUL.getItemValue() :
                    EmailTaskStatusEnum.PARTIAL_FAILURE.getItemValue());
            iEmailTaskService.saveOrUpdate(emailTask);

            // 记录任务完成日志
            log.info("邮件发送任务完成: 成功{}个, 失败{}个", successCount, failCount);

            // 设置任务执行结果
            if (failCount == 0) {
                log.info("发送完成: 成功" + successCount + "个");
            } else {
                log.info("发送完成: 成功" + successCount + "个, 失败" + failCount + "个");
            }

        } catch (Exception e) {
            // 任务执行过程中发生异常
            log.error("邮件发送任务执行异常", e);
            // 更新任务状态为全部发送失败
            emailTask.setStatus(EmailTaskStatusEnum.ALL_FAILED.getItemValue());
            iEmailTaskService.saveOrUpdate(emailTask);
            // 返回任务执行异常信息
            log.error("任务执行异常: " + e.getMessage());
            throw new BusinessException("立即发送邮件异常");
        }
        return Result.success();
    }

    /**
     * 测试发送邮件
     * @param request
     * @return
     */
    @Override
    public Result<ApiSendEmailResponse> testSendEmail(ApiSendEmailRequest request) {
        return null;
    }


}
