package com.yd.insurance.base.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.insurance.base.api.service.ApiInsuranceAdditionalProductService;
import com.yd.insurance.base.api.service.ApiRelInsuranceAdditionalProductDeptService;
import com.yd.insurance.base.api.service.ApiRelInsuranceAdditionalProductService;
import com.yd.insurance.base.feign.dto.CdDeptDto;
import com.yd.insurance.base.feign.dto.SsDeptDto;
import com.yd.insurance.base.feign.request.ApiInsuranceAdditionalProductAddRequest;
import com.yd.insurance.base.feign.request.ApiInsuranceAdditionalProductEditRequest;
import com.yd.insurance.base.feign.request.ApiInsuranceAdditionalProductPageRequest;
import com.yd.insurance.base.feign.response.ApiInsuranceAdditionalProductDetailResponse;
import com.yd.insurance.base.feign.response.ApiInsuranceAdditionalProductPageResponse;
import com.yd.insurance.base.service.dto.InsuranceAdditionalProductDto;
import com.yd.insurance.base.service.enums.ProductDeptEnum;
import com.yd.insurance.base.service.model.InsuranceAdditionalProduct;
import com.yd.insurance.base.service.service.IInsuranceAdditionalProductService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiInsuranceAdditionalProductServiceImpl implements ApiInsuranceAdditionalProductService {

    @Autowired
    private IInsuranceAdditionalProductService iInsuranceAdditionalProductService;

    @Autowired
    private ApiRelInsuranceAdditionalProductDeptService apiRelInsuranceAdditionalProductDeptService;

    /**
     * 保险附加产品分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiInsuranceAdditionalProductPageResponse>> page(ApiInsuranceAdditionalProductPageRequest request) {
        Page<ApiInsuranceAdditionalProductPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiInsuranceAdditionalProductPageResponse> iPage = iInsuranceAdditionalProductService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 获取保险附加产品详情
     * @param additionalProductBizId 保险附加产品唯一标识(业务ID)
     * @return
     */
    @Override
    public Result<ApiInsuranceAdditionalProductDetailResponse> detail(String additionalProductBizId) {
        Result<InsuranceAdditionalProduct> result = checkInsuranceAdditionalProductIsExist(additionalProductBizId);
        InsuranceAdditionalProduct insuranceAdditionalProduct = result.getData();

        ApiInsuranceAdditionalProductDetailResponse response = new ApiInsuranceAdditionalProductDetailResponse();
        BeanUtils.copyProperties(insuranceAdditionalProduct,response);

        if (StringUtils.isNotBlank(response.getPaymentTerm())) {
            //绑定供款年期列表
            List<String> paymentTermList = Arrays.stream(response.getPaymentTerm().split(","))
                    .collect(Collectors.toList());

            response.setPaymentTermList(paymentTermList);
        }

        ApiInsuranceAdditionalProductDetailResponse response1 = detailExpand(additionalProductBizId);
        response.setSsDeptDtoList(response1.getSsDeptDtoList());
        response.setCdDeptDtoList(response1.getCdDeptDtoList());

        return Result.success(response);
    }

    /**
     * 添加保险附加产品
     * @param request
     * @return
     */
    @Override
    public Result add(ApiInsuranceAdditionalProductAddRequest request) {

        //校验保险附加产品名称的唯一性
        List<InsuranceAdditionalProduct> list = iInsuranceAdditionalProductService.queryList(InsuranceAdditionalProductDto.builder().productName(request.getProductName()).build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.ADDITIONAL_PRODUCT_NAME_EXISTS.getCode(),ResultCode.ADDITIONAL_PRODUCT_NAME_EXISTS.getMessage());
        }
        //新增保险附加产品
        InsuranceAdditionalProduct insuranceAdditionalProduct = new InsuranceAdditionalProduct();
        BeanUtils.copyProperties(request,insuranceAdditionalProduct);
        insuranceAdditionalProduct.setAdditionalProductBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_INSURANCE_ADDITIONAL_PRODUCT.getCode()));

        if (!CollectionUtils.isEmpty(request.getPaymentTermList())) {
            //绑定供款年期列表转成逗号分隔的字符串
            insuranceAdditionalProduct.setPaymentTerm(String.join(",", request.getPaymentTermList()));
        }
        iInsuranceAdditionalProductService.saveOrUpdate(insuranceAdditionalProduct);

        //更新绑定附加产品所属保险公司唯一业务ID列表（部门id集合）
        apiRelInsuranceAdditionalProductDeptService.updateRelInsuranceAdditionalProductDeptList(ProductDeptEnum.SS.getValue(),
                insuranceAdditionalProduct.getAdditionalProductBizId(),
                request.getSsDeptBizIdList());

        //更新绑定的保险附加产品出单公司列表数据（部门id集合）
        apiRelInsuranceAdditionalProductDeptService.updateRelInsuranceAdditionalProductDeptList(ProductDeptEnum.CD.getValue(),
                insuranceAdditionalProduct.getAdditionalProductBizId(),
                request.getCdDeptBizIdList());

        return Result.success();
    }

    /**
     * 编辑保险附加产品
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiInsuranceAdditionalProductEditRequest request) {

        //校验保险附加产品是否存在
        Result<InsuranceAdditionalProduct> result = checkInsuranceAdditionalProductIsExist(request.getAdditionalProductBizId());
        InsuranceAdditionalProduct insuranceAdditionalProduct = result.getData();

        //校验保险附加产品名称的唯一性
        List<InsuranceAdditionalProduct> list = iInsuranceAdditionalProductService.queryList(InsuranceAdditionalProductDto.builder()
                .productName(request.getProductName())
                .additionalProductBizId(request.getAdditionalProductBizId())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.ADDITIONAL_PRODUCT_NAME_EXISTS.getCode(),ResultCode.ADDITIONAL_PRODUCT_NAME_EXISTS.getMessage());
        }

        //编辑保险附加产品
        BeanUtils.copyProperties(request,insuranceAdditionalProduct);

        if (!CollectionUtils.isEmpty(request.getPaymentTermList())) {
            //绑定供款年期列表转成逗号分隔的字符串
            insuranceAdditionalProduct.setPaymentTerm(String.join(",", request.getPaymentTermList()));
        }
        iInsuranceAdditionalProductService.saveOrUpdate(insuranceAdditionalProduct);

        //更新绑定附加产品所属保险公司唯一业务ID列表（部门id集合）
        apiRelInsuranceAdditionalProductDeptService.updateRelInsuranceAdditionalProductDeptList(ProductDeptEnum.SS.getValue(),
                insuranceAdditionalProduct.getAdditionalProductBizId(),
                request.getSsDeptBizIdList());

        //更新绑定的保险附加产品出单公司列表数据（部门id集合）
        apiRelInsuranceAdditionalProductDeptService.updateRelInsuranceAdditionalProductDeptList(ProductDeptEnum.CD.getValue(),
                insuranceAdditionalProduct.getAdditionalProductBizId(),
                request.getCdDeptBizIdList());

        return Result.success();
    }

    /**
     * 删除保险附加产品（逻辑删除）
     * @param additionalProductBizId 保险附加产品唯一标识(业务ID)
     * @return
     */
    @Override
    public Result del(String additionalProductBizId) {
        return null;
    }

    /**
     * 编辑保险附加产品状态
     * @param additionalProductBizId 保险附加产品唯一标识(业务ID)
     * @param status '保险附加产品状态：0=下架，1=在售'
     * @return
     */
    @Override
    public Result editStatus(String additionalProductBizId, Integer status) {
        //校验保险附加产品是否存在
        Result<InsuranceAdditionalProduct> result = checkInsuranceAdditionalProductIsExist(additionalProductBizId);
        InsuranceAdditionalProduct insuranceAdditionalProduct = result.getData();

        insuranceAdditionalProduct.setProductStatus(status);
        iInsuranceAdditionalProductService.saveOrUpdate(insuranceAdditionalProduct);
        return Result.success();
    }

    /**
     * 获取保险附加产品详情-扩展信息
     * @param additionalProductBizId
     * @return
     */
    public ApiInsuranceAdditionalProductDetailResponse detailExpand(String additionalProductBizId) {
        ApiInsuranceAdditionalProductDetailResponse response = new ApiInsuranceAdditionalProductDetailResponse();
        //根据保险附加产品id查询所属公司对象集合
        Result<List<SsDeptDto>> ssResult = apiRelInsuranceAdditionalProductDeptService.getSsDeptDtoList(additionalProductBizId);
        //根据保险附加产品id查询出单公司对象集合
        Result<List<CdDeptDto>> cdResult = apiRelInsuranceAdditionalProductDeptService.getCdDeptDtoList(additionalProductBizId);

        response.setSsDeptDtoList(ssResult.getData());
        response.setCdDeptDtoList(cdResult.getData());

        return response;
    }

    /**
     * 校验保险附加产品是否存在
     * @param additionalProductBizId
     * @return
     */
    public Result<InsuranceAdditionalProduct> checkInsuranceAdditionalProductIsExist(String additionalProductBizId) {
        InsuranceAdditionalProduct insuranceAdditionalProduct = iInsuranceAdditionalProductService.queryOne(additionalProductBizId);
        if (Objects.isNull(insuranceAdditionalProduct)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(insuranceAdditionalProduct);
    }
}
