package com.yd.insurance.base.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.insurance.base.api.service.*;
import com.yd.insurance.base.feign.dto.CdDeptDto;
import com.yd.insurance.base.feign.dto.InsuranceAdditionalProductDto;
import com.yd.insurance.base.feign.dto.PlanDto;
import com.yd.insurance.base.feign.dto.SsDeptDto;
import com.yd.insurance.base.feign.request.ApiInsuranceProductAddRequest;
import com.yd.insurance.base.feign.request.ApiInsuranceProductEditRequest;
import com.yd.insurance.base.feign.request.ApiInsuranceProductPageRequest;
import com.yd.insurance.base.feign.response.ApiInsuranceProductDetailResponse;
import com.yd.insurance.base.feign.response.ApiInsuranceProductPageResponse;
import com.yd.insurance.base.service.dto.InsuranceProductDto;
import com.yd.insurance.base.service.enums.ProductDeptEnum;
import com.yd.insurance.base.service.model.InsuranceProduct;
import com.yd.insurance.base.service.service.IInsuranceProductService;
import com.yd.user.service.model.SysProject;
import com.yd.user.service.model.SysTenant;
import com.yd.user.service.service.ISysProjectService;
import com.yd.user.service.service.ISysTenantService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
public class ApiInsuranceProductServiceImpl implements ApiInsuranceProductService {

    @Autowired
    private IInsuranceProductService iInsuranceProductService;

    @Autowired
    private ApiRelTenantInsuranceProductService apiRelTenantInsuranceProductService;

    @Autowired
    private ApiRelInsuranceProductPlanService apiRelInsuranceProductPlanService;

    @Autowired
    private ApiRelInsuranceProductDeptService apiRelInsuranceProductDeptService;

    @Autowired
    private ApiRelInsuranceAdditionalProductService apiRelInsuranceAdditionalProductService;

    @Autowired
    private ISysTenantService iSysTenantService;

    @Autowired
    private ISysProjectService iSysProjectService;

    /**
     * 保险产品分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiInsuranceProductPageResponse>> page(ApiInsuranceProductPageRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的保险产品类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;
        //当前所属的租户id（当前页面右上角显示的租户） 超级管理员不传值，非超级管理员根据传值来
        if (!isAdmin && StringUtils.isBlank(request.getLoginTenantBizId())) {
            //非超级管理员并且LoginTenantBizId为空，提示必须传值LoginTenantBizId
            throw new BusinessException("非超级管理员，当前登录用户所属的租户id（当前页面右上角显示的租户）必须传值");
        }
        String loginTenantBizId = isAdmin ? "" : request.getLoginTenantBizId();

        Page<ApiInsuranceProductPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        List<String> projectBizIdList = new ArrayList<>();

        //TODO
//        if (!isAdmin) {
//            //查询当前租户下绑定的项目列表id
//            List<RelTenantProject> relTenantProjectList = iRelTenantProjectService.queryList(RelTenantProjectDto.builder().tenantBizId(loginTenantBizId).build());
//            if (CollectionUtils.isEmpty(relTenantProjectList)) {
//                projectBizIdList = relTenantProjectList.stream().map(RelTenantProject::getProjectBizId).collect(Collectors.toList());
//            }
//        }

        IPage<ApiInsuranceProductPageResponse> iPage = iInsuranceProductService.page(page, request,isAdmin,projectBizIdList,loginTenantBizId);
        return Result.success(iPage);
    }

    /**
     * 获取保险产品详情
     * @param productBizId 保险产品唯一标识(业务ID)
     * @return
     */
    @Override
    public Result<ApiInsuranceProductDetailResponse> detail(String productBizId) {
        Result<InsuranceProduct> result = checkInsuranceProductIsExist(productBizId);
        InsuranceProduct insuranceProduct = result.getData();

        ApiInsuranceProductDetailResponse response = new ApiInsuranceProductDetailResponse();
        BeanUtils.copyProperties(insuranceProduct,response);

        if (CommonEnum.SCOPE_SYS.getCode().equals(response.getScope().toString())) {
            //系统级
            response.setTenantBizId("");
            response.setTenantName("");
            response.setProjectBizId("");
            response.setProjectName("");
        }else if (CommonEnum.SCOPE_TENANT.getCode().equals(response.getScope().toString())) {
            //租户级
            SysTenant sysTenant = iSysTenantService.queryOne(response.getTenantBizId());
            if (!Objects.isNull(sysTenant)) {
                response.setTenantName(sysTenant.getTenantName());
            }
            response.setProjectBizId("");
            response.setProjectName("");
        }else if (CommonEnum.SCOPE_PROJECT.getCode().equals(response.getScope().toString())) {
            //项目级
            SysProject sysProject = iSysProjectService.queryOne(response.getProjectBizId());
            if (!Objects.isNull(sysProject)) {
                response.setProjectName(sysProject.getProjectName());
            }
            response.setTenantBizId("");
            response.setTenantName("");
        }

        if (StringUtils.isNotBlank(response.getPaymentTerm())) {
            //绑定供款年期列表
            List<String> paymentTermList = Arrays.stream(response.getPaymentTerm().split(","))
                    .collect(Collectors.toList());

            response.setPaymentTermList(paymentTermList);
        }

        ApiInsuranceProductDetailResponse response1 = detailExpand(productBizId);
        response.setSsDeptDtoList(response1.getSsDeptDtoList());
        response.setCdDeptDtoList(response1.getCdDeptDtoList());
        response.setPlanDtoList(response1.getPlanDtoList());
        response.setInsuranceAdditionalProductDtoList(response1.getInsuranceAdditionalProductDtoList());

        return Result.success(response);
    }

    /**
     * 获取保险产品详情-扩展信息
     * @param productBizId
     * @return
     */
    public ApiInsuranceProductDetailResponse detailExpand(String productBizId) {
        ApiInsuranceProductDetailResponse response = new ApiInsuranceProductDetailResponse();
        //根据产品id查询所属公司对象集合
        Result<List<SsDeptDto>> ssResult = apiRelInsuranceProductDeptService.getSsDeptDtoList(productBizId);
        //根据产品id查询出单公司对象集合
        Result<List<CdDeptDto>> cdResult = apiRelInsuranceProductDeptService.getCdDeptDtoList(productBizId);
        //根据产品id获取计划集合
        Result<List<PlanDto>> planResult = apiRelInsuranceProductPlanService.getPlanDtoList(productBizId);
        //根据产品id获取保险附加产品集合
        Result<List<InsuranceAdditionalProductDto>> insuranceAdditionalProductDtoList = apiRelInsuranceAdditionalProductService.getInsuranceAdditionalProductDtoList(productBizId);

        response.setSsDeptDtoList(ssResult.getData());
        response.setCdDeptDtoList(cdResult.getData());
        response.setPlanDtoList(planResult.getData());
        response.setInsuranceAdditionalProductDtoList(insuranceAdditionalProductDtoList.getData());

        return response;
    }

    /**
     * 添加保险产品
     * @param request
     * @return
     */
    @Override
    public Result add(ApiInsuranceProductAddRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的保险产品类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;

        //添加和编辑校验入参
        addAndEditReqCheck(isAdmin,
                request.getScope(),
                request.getTenantBizId(),
                request.getProjectBizId());

        //校验产品名称的唯一性
        List<InsuranceProduct> list = iInsuranceProductService.queryList(InsuranceProductDto.builder().productName(request.getProductName()).build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.PRODUCT_NAME_EXISTS.getCode(),ResultCode.PRODUCT_NAME_EXISTS.getMessage());
        }
        //新增保险产品
        InsuranceProduct insuranceProduct = new InsuranceProduct();
        BeanUtils.copyProperties(request,insuranceProduct);
        insuranceProduct.setProductBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_INSURANCE_PRODUCT.getCode()));
        insuranceProduct = addAndEditSaveSetCommon(insuranceProduct);

        if (!CollectionUtils.isEmpty(request.getPaymentTermList())) {
            //绑定供款年期列表转成逗号分隔的字符串
            insuranceProduct.setPaymentTerm(String.join(",", request.getPaymentTermList()));
        }
        iInsuranceProductService.saveOrUpdate(insuranceProduct);

        //新增或者编辑-更新租户校色关系、更新项目保险产品关系公共方法
        addAndEditByRelCommon(request.getTenantBizId(),
                request.getProjectBizId(),
                insuranceProduct.getProductBizId(),
                request.getScope(),
                CommonEnum.OPR_SOURCE_ADD.getCode());

        //更新绑定产品所属保险公司唯一业务ID列表（部门id集合）
        apiRelInsuranceProductDeptService.updateRelInsuranceProductDeptList(ProductDeptEnum.SS.getValue(),
                insuranceProduct.getProductBizId(),
                request.getSsDeptBizIdList());

        //更新绑定的保险产品出单公司列表数据（部门id集合）
        apiRelInsuranceProductDeptService.updateRelInsuranceProductDeptList(ProductDeptEnum.CD.getValue(),
                insuranceProduct.getProductBizId(),
                request.getCdDeptBizIdList());

        //更新绑定的保险产品计划列表数据
        apiRelInsuranceProductPlanService.updateRelInsuranceProductPlanList(insuranceProduct.getProductBizId(),request.getPlanBizIdList());

        //更新绑定的保险附加产品列表数据
        apiRelInsuranceAdditionalProductService.updateRelInsuranceAdditionalProductList(insuranceProduct.getProductBizId(),request.getAdditionalProductBizIdList());

        return Result.success();
    }

    /**
     * 编辑保险产品
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiInsuranceProductEditRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的保险产品类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;

        //添加和编辑校验入参
        addAndEditReqCheck(isAdmin,
                request.getScope(),
                request.getTenantBizId(),
                request.getProjectBizId());

        //校验保险产品是否存在
        Result<InsuranceProduct> result = checkInsuranceProductIsExist(request.getProductBizId());
        InsuranceProduct insuranceProduct = result.getData();

        //校验保险产品名称的唯一性
        List<InsuranceProduct> list = iInsuranceProductService.queryList(InsuranceProductDto.builder()
                .productName(request.getProductName())
                .productBizId(request.getProductBizId())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.PRODUCT_NAME_EXISTS.getCode(),ResultCode.PRODUCT_NAME_EXISTS.getMessage());
        }

        //TODO 换绑逻辑（作用域换绑：提示校验需要先把什么解绑再来换绑）

        //编辑保险产品
        BeanUtils.copyProperties(request,insuranceProduct);
        insuranceProduct = addAndEditSaveSetCommon(insuranceProduct);

        if (!CollectionUtils.isEmpty(request.getPaymentTermList())) {
            //绑定供款年期列表转成逗号分隔的字符串
            insuranceProduct.setPaymentTerm(String.join(",", request.getPaymentTermList()));
        }
        iInsuranceProductService.saveOrUpdate(insuranceProduct);

        //新增或者编辑-更新租户校色关系、更新项目保险产品关系公共方法 TODO
//        addAndEditByRelCommon(request.getTenantBizId(),
//                request.getProjectBizId(),
//                sysRole.getRoleBizId(),
//                request.getScope(),
//                CommonEnum.OPR_SOURCE_ADD.getCode());

        //更新绑定产品所属保险公司唯一业务ID列表（部门id集合）
        apiRelInsuranceProductDeptService.updateRelInsuranceProductDeptList(ProductDeptEnum.SS.getValue(),
                insuranceProduct.getProductBizId(),
                request.getSsDeptBizIdList());

        //更新绑定的保险产品出单公司列表数据（部门id集合）
        apiRelInsuranceProductDeptService.updateRelInsuranceProductDeptList(ProductDeptEnum.CD.getValue(),
                insuranceProduct.getProductBizId(),
                request.getCdDeptBizIdList());

        //更新绑定的保险产品计划列表数据
        apiRelInsuranceProductPlanService.updateRelInsuranceProductPlanList(insuranceProduct.getProductBizId(),request.getPlanBizIdList());

        //更新绑定的保险附加产品列表数据
        apiRelInsuranceAdditionalProductService.updateRelInsuranceAdditionalProductList(insuranceProduct.getProductBizId(),request.getAdditionalProductBizIdList());

        return Result.success();
    }

    /**
     * 删除保险产品（逻辑删除）
     * @param productBizId 保险产品唯一标识(业务ID)
     * @return
     */
    @Override
    public Result del(String productBizId) {
        return null;
    }

    /**
     * 编辑保险产品状态
     * @param productBizId 保险产品唯一标识(业务ID)
     * @param status '保险产品状态：0=下架，1=在售'
     * @return
     */
    @Override
    public Result editStatus(String productBizId,Integer status) {
        //校验保险产品是否存在
        Result<InsuranceProduct> result = checkInsuranceProductIsExist(productBizId);
        InsuranceProduct insuranceProduct = result.getData();

        insuranceProduct.setProductStatus(status);
        iInsuranceProductService.saveOrUpdate(insuranceProduct);
        return Result.success();
    }

    /**
     * 新增或者编辑-更新租户保险产品关系、更新项目保险产品关系公共方法
     * @param tenantBizId 租户id
     * @param projectBizId 项目id
     * @param productBizId 保险产品id
     * @param scope 作用域
     * @param oprSource 操作来源 1-添加 2-编辑
     */
    public void addAndEditByRelCommon(String tenantBizId,
                                      String projectBizId,
                                      String productBizId,
                                      Integer scope,
                                      String oprSource) {
        if (CommonEnum.OPR_SOURCE_ADD.getCode().equals(oprSource)) {
            //添加来源
            if (CommonEnum.SCOPE_TENANT.getCode().equals(scope.toString())) {
                //作用域租户级-新增租户保险产品关系
                apiRelTenantInsuranceProductService.addSingleRel(tenantBizId,productBizId);
            }
            //TODO
//            if (CommonEnum.SCOPE_PROJECT.getCode().equals(scope.toString())) {
//                //作用域项目级-新增项目保险产品关系
//                apiRelProjectRoleService.addSingleRel(projectBizId,productBizId);
//            }
        }else if (CommonEnum.OPR_SOURCE_EDIT.getCode().equals(oprSource)) {
            //编辑来源 TODO 编辑
        }
    }
    
    /**
     * 添加和编辑保存设置参数公共方法
     * @param insuranceProduct
     * @return
     */
    public InsuranceProduct addAndEditSaveSetCommon(InsuranceProduct insuranceProduct) {
        if (CommonEnum.SCOPE_SYS.getCode().equals(insuranceProduct.getScope().toString())) {
            //系统级
            insuranceProduct.setTenantBizId("");
            insuranceProduct.setProjectBizId("");
        }else if (CommonEnum.SCOPE_TENANT.getCode().equals(insuranceProduct.getScope().toString())) {
            //租户级
            insuranceProduct.setProjectBizId("");
        }else if (CommonEnum.SCOPE_PROJECT.getCode().equals(insuranceProduct.getScope().toString())) {
            //项目级
            insuranceProduct.setTenantBizId("");
        }
        return insuranceProduct;
    }

    /**
     * 添加和编辑校验入参
     * @param isAdmin
     * @param scope
     * @param tenantBizId
     * @param projectBizId
     */
    public void addAndEditReqCheck(Boolean isAdmin,
                                   Integer scope,
                                   String tenantBizId,
                                   String projectBizId) {
        if (!isAdmin && CommonEnum.SCOPE_SYS.getCode().equals(scope.toString())) {
            //非超级管理员不能选择系统级作用域
            throw new BusinessException("非超级管理员不能选择系统级作用域！");
        }
        //校验作用域scope字段，如果为作用域租户级所属租户id必须传值
        if(CommonEnum.SCOPE_TENANT.getCode().equals(scope.toString())
                && StringUtils.isBlank(tenantBizId)) {
            throw new BusinessException("租户级作用域，所属租户id不能为空");
        }
        //校验作用域scope字段，如果为作用域项目级所属项目id必须传值
        if(CommonEnum.SCOPE_PROJECT.getCode().equals(scope.toString())
                && StringUtils.isBlank(projectBizId)) {
            throw new BusinessException("项目级作用域，所属项目id不能为空");
        }
    }

    /**
     * 校验保险产品是否存在
     * @param productBizId
     * @return
     */
    public Result<InsuranceProduct> checkInsuranceProductIsExist(String productBizId) {
        InsuranceProduct insuranceProduct = iInsuranceProductService.queryOne(productBizId);
        if (Objects.isNull(insuranceProduct)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(insuranceProduct);
    }
}
