package com.yd.insurance.base.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.insurance.base.api.service.ApiInsuranceCompanyService;
import com.yd.insurance.base.feign.dto.ApiInsuranceCompanyDto;
import com.yd.insurance.base.feign.request.insurancecompany.ApiInsuranceCompanyAddRequest;
import com.yd.insurance.base.feign.request.insurancecompany.ApiInsuranceCompanyEditRequest;
import com.yd.insurance.base.feign.request.insurancecompany.ApiInsuranceCompanyPageRequest;
import com.yd.insurance.base.feign.response.insurancecompany.ApiInsuranceCompanyDetailResponse;
import com.yd.insurance.base.feign.response.insurancecompany.ApiInsuranceCompanyPageResponse;
import com.yd.insurance.base.service.dto.InsuranceCompanyDto;
import com.yd.insurance.base.service.dto.InsuranceProductPlanDto;
import com.yd.insurance.base.service.model.InsuranceCompany;
import com.yd.insurance.base.service.model.InsuranceProductPlan;
import com.yd.insurance.base.service.service.IInsuranceCompanyService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;

@Slf4j
@Service
public class ApiInsuranceCompanyServiceImpl implements ApiInsuranceCompanyService {

    @Autowired
    private IInsuranceCompanyService iInsuranceCompanyService;

    /**
     * 分页列表查询-保险公司信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiInsuranceCompanyPageResponse>> page(ApiInsuranceCompanyPageRequest request) {
        Page<ApiInsuranceCompanyPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiInsuranceCompanyPageResponse> iPage = iInsuranceCompanyService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 详情-保险公司信息
     * @param insuranceCompanyBizId 保险公司唯一业务ID
     * @return
     */
    @Override
    public Result<ApiInsuranceCompanyDetailResponse> detail(String insuranceCompanyBizId) {
        //校验保险公司信息是否存在
        Result<InsuranceCompany> result = checkInsuranceCompany(insuranceCompanyBizId);
        InsuranceCompany insuranceCompany = result.getData();
        ApiInsuranceCompanyDetailResponse response = new ApiInsuranceCompanyDetailResponse();
        //保险公司主信息
        ApiInsuranceCompanyDto apiInsuranceCompanyDto = new ApiInsuranceCompanyDto();
        BeanUtils.copyProperties(insuranceCompany,apiInsuranceCompanyDto);
        response.setApiInsuranceCompanyDto(apiInsuranceCompanyDto);
        return Result.success(response);
    }

    /**
     * 添加-保险公司信息
     * @param request
     * @return
     */
    @Override
    public Result add(ApiInsuranceCompanyAddRequest request) {
        //添加-保险公司主信息
        addInsuranceCompanyMain(request.getApiInsuranceCompanyDto());

        return Result.success();
    }

    /**
     * 添加-保险公司主信息
     * @param apiInsuranceCompanyDto
     * @return
     */
    public Result addInsuranceCompanyMain(ApiInsuranceCompanyDto apiInsuranceCompanyDto) {
        //校验保险公司全称的唯一性
        List<InsuranceCompany> list = iInsuranceCompanyService.queryList(InsuranceCompanyDto.builder()
                .fullName(apiInsuranceCompanyDto.getFullName())
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("保险公司全称已存在");
        }
        InsuranceCompany insuranceCompany = new InsuranceCompany();
        BeanUtils.copyProperties(apiInsuranceCompanyDto,insuranceCompany);
        //保险公司表唯一业务ID
        insuranceCompany.setInsuranceCompanyBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_INSURANCE_COMPANY.getCode()));
        iInsuranceCompanyService.saveOrUpdate(insuranceCompany);
        return Result.success();
    }

    /**
     * 编辑-保险公司信息
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiInsuranceCompanyEditRequest request) {
        //编辑-保险公司主信息
        editInsuranceCompanyMain(request.getApiInsuranceCompanyDto());
        return Result.success();
    }

    /**
     * 编辑-保险公司主信息
     * @param dto
     * @return
     */
    public Result editInsuranceCompanyMain(ApiInsuranceCompanyDto dto) {
        if (StringUtils.isBlank(dto.getInsuranceCompanyBizId())) {
            throw new BusinessException("保险公司表唯一业务ID不能为空");
        }
        //校验保险公司信息是否存在
        Result<InsuranceCompany> result = checkInsuranceCompany(dto.getInsuranceCompanyBizId());
        InsuranceCompany insuranceCompany = result.getData();

        //校验保险公司全称的唯一性
        List<InsuranceCompany> list = iInsuranceCompanyService.queryList(InsuranceCompanyDto.builder()
                .fullName(dto.getFullName())
                .insuranceCompanyBizId(dto.getInsuranceCompanyBizId())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("保险公司全称已存在");
        }
        BeanUtils.copyProperties(dto,insuranceCompany);
        iInsuranceCompanyService.saveOrUpdate(insuranceCompany);
        return Result.success();
    }

    /**
     * 删除-保险公司信息
     * @param insuranceCompanyBizId 保险公司唯一业务ID
     * @return
     */
    @Override
    public Result del(String insuranceCompanyBizId) {
        return null;
    }

    /**
     * 编辑-保险公司信息合作状态
     * @param insuranceCompanyBizId 保险公司唯一业务ID
     * @param cooperationStatus 合作状态（合作中、已终止、未合作）
     * @return
     */
    @Override
    public Result editStatus(String insuranceCompanyBizId, String cooperationStatus) {
        //校验保险公司信息是否存在
        Result<InsuranceCompany> result = checkInsuranceCompany(insuranceCompanyBizId);
        InsuranceCompany insuranceCompany = result.getData();
        insuranceCompany.setCooperationStatus(cooperationStatus);
        iInsuranceCompanyService.saveOrUpdate(insuranceCompany);
        return Result.success();
    }

    /**
     * 校验保险公司信息是否存在
     * @param insuranceCompanyBizId
     * @return
     */
    public Result<InsuranceCompany> checkInsuranceCompany(String insuranceCompanyBizId) {
        InsuranceCompany insuranceCompany = iInsuranceCompanyService.queryOne(insuranceCompanyBizId);
        if (Objects.isNull(insuranceCompany)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(insuranceCompany);
    }
}
