package com.yd.oss.feign.client;

import com.yd.common.result.Result;
import com.yd.oss.feign.fallback.ApiOssFeignFallbackFactory;
import com.yd.oss.feign.request.ApiUploadFileRequest;
import com.yd.oss.feign.response.ApiFileMetadataResponse;
import com.yd.oss.feign.response.ApiUploadResponse;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

/**
 * OSS服务信息Feign客户端
 */
@FeignClient(name = "yd-oss-api", fallbackFactory = ApiOssFeignFallbackFactory.class)
public interface ApiOssFeignClient {

    /**
     * 上传文件
     * @param file 上传的文件
     * @return
     */
    @PostMapping("/upload")
    Result<ApiUploadResponse> uploadFile(@RequestParam("file") MultipartFile file);

    /**
     * 上传文件 - body入参方式（新，推荐）
     * @param file
     * @param request
     * @return
     */
    @PostMapping(value = "/upload/body", consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    Result<ApiUploadResponse> uploadFileBody(@RequestPart("file") MultipartFile file, ApiUploadFileRequest request);

    /**
     * 下载文件
     * @param fileKey 文件唯一标识
     * @return
     */
    @GetMapping("/download")
    Result downloadFile(@RequestParam("fileKey") String fileKey);

    /**
     * 删除文件
     * @param fileKey 文件唯一标识
     * @return
     */
    @DeleteMapping("/delete")
    Result<Boolean> deleteFile(@RequestParam("fileKey") String fileKey);

    /**
     * 获取文件URL
     * @param fileKey 文件唯一标识
     * @param expirySeconds 文件URL过期秒数
     * @return
     */
    @GetMapping("/url")
    Result<String> getFileUrl(@RequestParam("fileKey") String fileKey,
                              @RequestParam(value = "expiry", defaultValue = "3600") Long expirySeconds);

    /**
     * 切换OSS提供商
     * @param providerName OSS提供商名称
     * @return
     */
    @PostMapping("/switch-provider")
    Result<Boolean> switchProvider(@RequestParam("providerName") String providerName);

    /**
     * 检查文件是否存在
     * @param fileKey 文件唯一标识
     * @return
     */
    @GetMapping("/exists")
    Result<Boolean> doesFileExist(@RequestParam("fileKey") String fileKey);

    /**
     * 获取文件元数据
     * @param fileKey 文件唯一标识
     * @return
     */
    @GetMapping("/metadata")
    Result<ApiFileMetadataResponse> getFileMetadata(@RequestParam("fileKey") String fileKey);

}
