package com.yd.oss.service.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.yd.common.exception.BusinessException;
import com.yd.oss.feign.response.ApiOssFileListResponse;
import com.yd.oss.service.dao.OssOperationLogMapper;
import com.yd.oss.service.dto.OssFileDto;
import com.yd.oss.service.model.OssFile;
import com.yd.oss.service.dao.OssFileMapper;
import com.yd.oss.service.model.OssOperationLog;
import com.yd.oss.service.service.IOssFileService;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.util.Date;
import java.util.List;

/**
 * <p>
 * 文件元数据表 服务实现类
 * </p>
 *
 * @author zxm
 * @since 2025-08-22
 */

@Slf4j
@Primary
@Service
public class OssFileServiceImpl extends ServiceImpl<OssFileMapper, OssFile> implements IOssFileService {

    @Resource
    private OssFileMapper ossFileMapper;

    @Resource
    private OssOperationLogMapper ossOperationLogMapper;

    /**
     * 保存文件元数据
     * @param ossFile
     * @return
     */
    @Override
    public boolean saveFileMetadata(OssFile ossFile) {
        if (ossFile.getId() == null) {
            // 新增
            ossFile.setUploadTime(LocalDateTime.now());
            ossFile.setIsDeleted(0);
            return ossFileMapper.insert(ossFile) > 0;
        } else {
            // 更新
            return ossFileMapper.updateById(ossFile) > 0;
        }
    }

    /**
     * 根据文件key获取文件
     * @param fileKey
     * @return
     */
    @Override
    public OssFile getFileByKey(String fileKey) {
        OssFile file = ossFileMapper.selectByFileKey(fileKey);
        if (file == null) {
            log.error("未找到文件: {}", fileKey);
            throw new BusinessException("未找到文件: " + fileKey);
        }
        return file;
    }

    /**
     * 标记文件为已删除
     * @param fileKey
     * @return
     */
    @Override
    public boolean markFileAsDeleted(String fileKey) {
        return ossFileMapper.markAsDeleted(fileKey) > 0;
    }

    /**
     * 保存操作日志
     * @param log
     * @return
     */
    @Override
    public boolean saveOperationLog(OssOperationLog log) {
        return ossOperationLogMapper.insert(log) > 0;
    }

    /**
     * 根据上传用户获取文件列表
     * @param uploadUser
     * @return
     */
    @Override
    public List<OssFile> getUserFiles(String uploadUser) {
        return ossFileMapper.selectByUploadUser(uploadUser);
    }

    /**
     * 获取所有未删除的文件
     * @return
     */
    @Override
    public List<OssFile> getActiveFiles() {
        return ossFileMapper.selectActiveFiles();
    }

    /**
     * 根据提供商ID获取文件列表
     * @param providerId
     * @return
     */
    @Override
    public List<OssFile> getProviderFiles(Long providerId) {
        return ossFileMapper.selectByProviderId(providerId);
    }

    /**
     * 根据文件ID获取操作日志
     * @param fileId
     * @return
     */
    @Override
    public List<OssOperationLog> getOperationLogsByFileId(Long fileId) {
        return ossOperationLogMapper.selectByFileId(fileId);
    }

    /**
     * 查询列表
     * @param dto
     * @return
     */
    @Override
    public List<OssFile> queryList(OssFileDto dto) {
        List<OssFile> list = baseMapper.selectList(new LambdaQueryWrapper<OssFile>()
                .eq(StringUtils.isNotBlank(dto.getObjectBizId()),OssFile::getObjectBizId,dto.getObjectBizId())
                .in(!CollectionUtils.isEmpty(dto.getFileBizIdList()),OssFile::getFileBizId,dto.getFileBizIdList())
        );
        return list;
    }

    @Override
    public OssFile queryOne(String fileBizId) {
        return this.getOne(new LambdaQueryWrapper<OssFile>().eq(OssFile::getFileBizId,fileBizId));
    }

    @Override
    public List<ApiOssFileListResponse> list(String objectBizId) {
        return this.baseMapper.list(objectBizId);
    }


}
