package com.yd.product.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.base.feign.enums.SpeciesTypeEnum;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.product.api.service.ApiExpectedSpeciesService;
import com.yd.product.api.service.ApiProductLaunchService;
import com.yd.product.api.service.ApiSpeciesPriceService;
import com.yd.product.api.service.ApiSpeciesSettingService;
import com.yd.product.feign.dto.ApiSpeciesPriceDto;
import com.yd.product.feign.request.expectedspecies.ApiExpectedSpeciesImportRequest;
import com.yd.product.feign.request.expectedspecies.ApiExpectedSpeciesPageRequest;
import com.yd.product.feign.response.expectedspecies.ApiExpectedSpeciesPageResponse;
import com.yd.product.service.model.AnnouncementSpecies;
import com.yd.product.service.model.ExpectedSpecies;
import com.yd.product.service.service.IExpectedCommissionRatioService;
import com.yd.product.service.service.IExpectedSpeciesService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiExpectedSpeciesServiceImpl implements ApiExpectedSpeciesService {

    @Autowired
    private IExpectedSpeciesService iExpectedSpeciesService;

    @Autowired
    private ApiProductLaunchService apiProductLaunchService;

    @Autowired
    private ApiSpeciesPriceService apiSpeciesPriceService;

    @Autowired
    private ApiSpeciesSettingService apiSpeciesSettingService;

    @Autowired
    private IExpectedCommissionRatioService iExpectedCommissionRatioService;

    /**
     * 分页列表查询-来佣比率规格信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiExpectedSpeciesPageResponse>> page(ApiExpectedSpeciesPageRequest request) {
        Page<ApiExpectedSpeciesPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiExpectedSpeciesPageResponse> iPage = iExpectedSpeciesService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 导入规格数据
     * @param request
     * @return
     */
    @Override
    public Result importSpecies(ApiExpectedSpeciesImportRequest request) {
        //校验产品上架信息是否存在
        apiProductLaunchService.checkProductLaunchIsExist(request.getProductLaunchBizId());
        //查询规格价格配置列表
        List<ApiSpeciesPriceDto> apiSpeciesPriceDtoList = apiSpeciesPriceService.querySpeciesPriceDtoList(request.getProductLaunchBizId());
        if (CollectionUtils.isEmpty(apiSpeciesPriceDtoList)) {
            throw new BusinessException("导入的数据不存在");
        }
        List<ExpectedSpecies> expectedSpeciesList = iExpectedSpeciesService.queryList(request.getProductLaunchBizId());
        if (!CollectionUtils.isEmpty(expectedSpeciesList)) {
            List<String> expectedSpeciesBizIdList = expectedSpeciesList.stream()
                    .map(ExpectedSpecies::getExpectedSpeciesBizId)
                    .collect(Collectors.toList());
            iExpectedCommissionRatioService.del(expectedSpeciesBizIdList);
        }
        iExpectedSpeciesService.del(request.getProductLaunchBizId());

        //查询规格价格配置列表 -> 新增来佣比率规格数据
        List<ExpectedSpecies> saveList = apiSpeciesPriceDtoList.stream().map(dto -> {
            ExpectedSpecies species = new ExpectedSpecies();
            //规格条件列表JSON串
            species.setSpeciesJson(dto.getSpeciesJson());
            //来佣比率规格表唯一业务ID
            species.setExpectedSpeciesBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EXPECTED_SPECIES.getCode()));
            //供款年期（5、10、20年期等）(固定规格条件)
            species.setPaymentTerm(apiSpeciesSettingService.getSpeciesValue(dto.getApiSpeciesSettingDtoList(), SpeciesTypeEnum.PAYMENT_TERM.getItemValue()));
            species.setProductLaunchBizId(request.getProductLaunchBizId());
            //保障计划（0-64岁等）(固定规格条件)
            species.setProtectionPlan(apiSpeciesSettingService.getSpeciesValue(dto.getApiSpeciesSettingDtoList(), SpeciesTypeEnum.PROTECTION_PLAN.getItemValue()));
            return species;
        }).collect(Collectors.toList());
        iExpectedSpeciesService.saveOrUpdateBatch(saveList);
        return Result.success();
    }

    /**
     * 删除规格数据
     * @param expectedSpeciesBizId
     * @return
     */
    @Override
    public Result delSpecies(String expectedSpeciesBizId) {
        return null;
    }

    /**
     * 校验来佣比率规格信息是否存在
     * @param expectedSpeciesBizId
     * @return
     */
    @Override
    public Result<ExpectedSpecies> checkExpectedSpeciesIsExist(String expectedSpeciesBizId) {
        ExpectedSpecies species = iExpectedSpeciesService.queryOne(expectedSpeciesBizId);
        if (Objects.isNull(species)) {
            //数据不存在
            throw new BusinessException("来佣比率规格信息不存在");
        }
        return Result.success(species);
    }
}
