package com.yd.product.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.product.api.service.ApiAnnouncementCommissionRatioService;
import com.yd.product.api.service.ApiAnnouncementSpeciesService;
import com.yd.product.feign.request.announcementcommissionratio.ApiAnnouncementCommissionRatioBatchSaveRequest;
import com.yd.product.feign.request.announcementcommissionratio.ApiAnnouncementCommissionRatioEditRequest;
import com.yd.product.feign.request.announcementcommissionratio.ApiAnnouncementCommissionRatioPageRequest;
import com.yd.product.feign.request.announcementcommissionratio.ApiAnnouncementCommissionRatioAddRequest;
import com.yd.product.feign.response.announcementcommissionratio.ApiAnnouncementCommissionRatioDetailResponse;
import com.yd.product.feign.response.announcementcommissionratio.ApiAnnouncementCommissionRatioPageResponse;
import com.yd.product.service.model.AnnouncementCommissionRatio;
import com.yd.product.service.service.IAnnouncementCommissionRatioService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiAnnouncementCommissionRatioServiceImpl implements ApiAnnouncementCommissionRatioService {

    @Autowired
    private IAnnouncementCommissionRatioService iAnnouncementCommissionRatioService;

    @Autowired
    private ApiAnnouncementSpeciesService apiAnnouncementSpeciesService;

    /**
     * 分页列表查询-公告佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiAnnouncementCommissionRatioPageResponse>> page(ApiAnnouncementCommissionRatioPageRequest request) {
        Page<ApiAnnouncementCommissionRatioPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiAnnouncementCommissionRatioPageResponse> iPage = iAnnouncementCommissionRatioService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 批量保存-公告佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result batchSave(ApiAnnouncementCommissionRatioBatchSaveRequest request) {
        List<String> announcementSpeciesBizIdList = new ArrayList<>();
        announcementSpeciesBizIdList.add(request.getAnnouncementSpeciesBizId());
        //先删后新增
        iAnnouncementCommissionRatioService.del(announcementSpeciesBizIdList);

        //新增
        if (!CollectionUtils.isEmpty(request.getRatioBatchSaveDtoList())) {
            List<AnnouncementCommissionRatio> saveList = request.getRatioBatchSaveDtoList()
                    .stream()
                    .map(dto -> {
                        AnnouncementCommissionRatio ratio = new AnnouncementCommissionRatio();
                        BeanUtils.copyProperties(dto,ratio);
                        ratio.setAnnouncementCommissionRatioBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_ANNOUNCEMENT_COMMISSION_RATIO.getCode()));
                        return ratio;
                    }).collect(Collectors.toList());
            iAnnouncementCommissionRatioService.saveOrUpdateBatch(saveList);
        }
        return Result.success();
    }

    /**
     * 添加-公告佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result add(ApiAnnouncementCommissionRatioAddRequest request) {
        //校验公告佣比率规格信息是否存在
        apiAnnouncementSpeciesService.checkAnnouncementSpeciesIsExist(request.getAnnouncementSpeciesBizId());
        //入参校验,佣金年限校验不同条件下的唯一性 TODO
        //新增
        AnnouncementCommissionRatio ratio = new AnnouncementCommissionRatio();
        BeanUtils.copyProperties(request,ratio);
        ratio.setAnnouncementCommissionRatioBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_ANNOUNCEMENT_COMMISSION_RATIO.getCode()));
        iAnnouncementCommissionRatioService.saveOrUpdate(ratio);
        return Result.success();
    }

    /**
     * 编辑-公告佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiAnnouncementCommissionRatioEditRequest request) {
        //校验公告佣比率规格明细信息是否存在
        Result<AnnouncementCommissionRatio> result = checkAnnouncementCommissionRatioIsExist(request.getAnnouncementCommissionRatioBizId());
        AnnouncementCommissionRatio ratio = result.getData();
        //校验公告佣比率规格信息是否存在
        apiAnnouncementSpeciesService.checkAnnouncementSpeciesIsExist(request.getAnnouncementSpeciesBizId());
        //入参校验,佣金年限校验不同条件下的唯一性 TODO
        //新增
        BeanUtils.copyProperties(request,ratio);
        iAnnouncementCommissionRatioService.saveOrUpdate(ratio);
        return Result.success();
    }

    /**
     * 详情-公告佣比率规格明细信息
     * @param announcementCommissionRatioBizId 公告佣比率规格明细表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiAnnouncementCommissionRatioDetailResponse> detail(String announcementCommissionRatioBizId) {
        //校验公告佣比率规格明细信息是否存在
        Result<AnnouncementCommissionRatio> result = checkAnnouncementCommissionRatioIsExist(announcementCommissionRatioBizId);
        AnnouncementCommissionRatio announcementCommissionRatio = result.getData();
        ApiAnnouncementCommissionRatioDetailResponse response = new ApiAnnouncementCommissionRatioDetailResponse();
        BeanUtils.copyProperties(announcementCommissionRatio,response);
        return Result.success(response);
    }

    /**
     * 删除-公告佣比率规格明细信息
     * @param announcementCommissionRatioBizId 公告佣比率规格明细表唯一业务ID
     * @return
     */
    @Override
    public Result del(String announcementCommissionRatioBizId) {
        //校验公告佣比率规格明细信息是否存在
        Result<AnnouncementCommissionRatio> result = checkAnnouncementCommissionRatioIsExist(announcementCommissionRatioBizId);
        AnnouncementCommissionRatio announcementCommissionRatio = result.getData();
        iAnnouncementCommissionRatioService.removeById(announcementCommissionRatio.getId());
        return Result.success();
    }

    /**
     * 复制-公告佣比率规格明细信息
     * @param announcementCommissionRatioBizId 公告佣比率规格明细表唯一业务ID
     * @return
     */
    @Override
    public Result copy(String announcementCommissionRatioBizId) {
        //校验公告佣比率规格明细信息是否存在
        Result<AnnouncementCommissionRatio> result = checkAnnouncementCommissionRatioIsExist(announcementCommissionRatioBizId);
        AnnouncementCommissionRatio announcementCommissionRatio = result.getData();
        announcementCommissionRatio.setId(null);
        announcementCommissionRatio.setAnnouncementCommissionRatioBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_ANNOUNCEMENT_COMMISSION_RATIO.getCode()));
        iAnnouncementCommissionRatioService.saveOrUpdate(announcementCommissionRatio);
        return Result.success();
    }

    /**
     * 校验公告佣比率规格明细信息是否存在
     * @param announcementCommissionRatioBizId
     * @return
     */
    @Override
    public Result<AnnouncementCommissionRatio> checkAnnouncementCommissionRatioIsExist(String announcementCommissionRatioBizId) {
        AnnouncementCommissionRatio announcementCommissionRatio = iAnnouncementCommissionRatioService.queryOne(announcementCommissionRatioBizId);
        if (Objects.isNull(announcementCommissionRatio)) {
            //数据不存在
            throw new BusinessException("公告佣比率规格明细信息不存在");
        }
        return Result.success(announcementCommissionRatio);
    }

}
