package com.yd.product.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.product.api.service.ApiExpectedCommissionRatioService;
import com.yd.product.api.service.ApiExpectedSpeciesService;
import com.yd.product.api.utils.ProductCommonUtils;
import com.yd.product.feign.dto.ApiExpectedCommissionRatioBatchSaveDto;
import com.yd.product.feign.dto.ApiExpectedCommissionRatioBatchSaveDto;
import com.yd.product.feign.request.expectedcommissionratio.ApiExpectedCommissionRatioAddRequest;
import com.yd.product.feign.request.expectedcommissionratio.ApiExpectedCommissionRatioBatchSaveRequest;
import com.yd.product.feign.request.expectedcommissionratio.ApiExpectedCommissionRatioEditRequest;
import com.yd.product.feign.request.expectedcommissionratio.ApiExpectedCommissionRatioPageRequest;
import com.yd.product.feign.response.expectedcommissionratio.ApiExpectedCommissionRatioDetailResponse;
import com.yd.product.feign.response.expectedcommissionratio.ApiExpectedCommissionRatioPageResponse;
import com.yd.product.service.model.AnnouncementCommissionRatio;
import com.yd.product.service.model.ExpectedCommissionRatio;
import com.yd.product.service.service.IExpectedCommissionRatioService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiExpectedCommissionRatioServiceImpl implements ApiExpectedCommissionRatioService {

    @Autowired
    private IExpectedCommissionRatioService iExpectedCommissionRatioService;

    @Autowired
    private ApiExpectedSpeciesService apiExpectedSpeciesService;

    /**
     * 分页列表查询-来佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiExpectedCommissionRatioPageResponse>> page(ApiExpectedCommissionRatioPageRequest request) {
        Page<ApiExpectedCommissionRatioPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiExpectedCommissionRatioPageResponse> iPage = iExpectedCommissionRatioService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 批量保存-来佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result batchSave(ApiExpectedCommissionRatioBatchSaveRequest request) {
        checkBatchSaveRequestPram(request.getRatioBatchSaveDtoList());
        List<String> expectedSpeciesBizIdList = new ArrayList<>();
        expectedSpeciesBizIdList.add(request.getExpectedSpeciesBizId());
        //先删后新增
        iExpectedCommissionRatioService.del(expectedSpeciesBizIdList);

        //新增
        if (!CollectionUtils.isEmpty(request.getRatioBatchSaveDtoList())) {
            List<ExpectedCommissionRatio> saveList = request.getRatioBatchSaveDtoList()
                    .stream()
                    .map(dto -> {
                        ExpectedCommissionRatio ratio = new ExpectedCommissionRatio();
                        BeanUtils.copyProperties(dto,ratio);
                        ratio.setExpectedSpeciesBizId(request.getExpectedSpeciesBizId());
                        ratio.setExpectedCommissionRatioBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EXPECTED_COMMISSION_RATIO.getCode()));
                        return ratio;
                    }).collect(Collectors.toList());
            iExpectedCommissionRatioService.saveOrUpdateBatch(saveList);
        }
        return Result.success();
    }

    /**
     * 添加-来佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result add(ApiExpectedCommissionRatioAddRequest request) {
        //校验来佣比率规格信息是否存在
        apiExpectedSpeciesService.checkExpectedSpeciesIsExist(request.getExpectedSpeciesBizId());
        //入参校验,佣金年限校验不同条件下的唯一性 TODO
        //新增
        ExpectedCommissionRatio ratio = new ExpectedCommissionRatio();
        BeanUtils.copyProperties(request,ratio);
        ratio.setExpectedCommissionRatioBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EXPECTED_COMMISSION_RATIO.getCode()));
        iExpectedCommissionRatioService.saveOrUpdate(ratio);
        return Result.success();
    }

    /**
     * 编辑-来佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiExpectedCommissionRatioEditRequest request) {
        //校验来佣比率规格明细信息是否存在
        Result<ExpectedCommissionRatio> result = checkExpectedCommissionRatioIsExist(request.getExpectedCommissionRatioBizId());
        ExpectedCommissionRatio ratio = result.getData();
        //校验来佣比率规格信息是否存在
        apiExpectedSpeciesService.checkExpectedSpeciesIsExist(request.getExpectedSpeciesBizId());
        //入参校验,佣金年限校验不同条件下的唯一性 TODO
        //编辑
        BeanUtils.copyProperties(request,ratio);
        iExpectedCommissionRatioService.saveOrUpdate(ratio);
        return Result.success();
    }

    /**
     * 编辑-状态
     * @param expectedCommissionRatioBizId
     * @param status
     * @return
     */
    @Override
    public Result editStatus(String expectedCommissionRatioBizId, Integer status) {
        //校验来佣比率规格明细信息是否存在
        Result<ExpectedCommissionRatio> result = checkExpectedCommissionRatioIsExist(expectedCommissionRatioBizId);
        ExpectedCommissionRatio ratio = result.getData();
        ratio.setStatus(status);
        iExpectedCommissionRatioService.saveOrUpdate(ratio);
        return Result.success();
    }

    /**
     * 详情-来佣比率规格明细信息
     * @param expectedSpeciesBizId 来佣比率规格明细表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiExpectedCommissionRatioDetailResponse> detail(String expectedSpeciesBizId) {
        //校验来佣比率规格明细信息是否存在
        Result<ExpectedCommissionRatio> result = checkExpectedCommissionRatioIsExist(expectedSpeciesBizId);
        ExpectedCommissionRatio ratio = result.getData();
        ApiExpectedCommissionRatioDetailResponse response = new ApiExpectedCommissionRatioDetailResponse();
        BeanUtils.copyProperties(ratio,response);
        return Result.success(response);
    }

    /**
     * 删除-来佣比率规格明细信息
     * @param expectedCommissionRatioBizId 来佣比率规格明细表唯一业务ID
     * @return
     */
    @Override
    public Result del(String expectedCommissionRatioBizId) {
        //校验来佣比率规格明细信息是否存在
        Result<ExpectedCommissionRatio> result = checkExpectedCommissionRatioIsExist(expectedCommissionRatioBizId);
        ExpectedCommissionRatio ratio = result.getData();
        iExpectedCommissionRatioService.removeById(ratio.getId());
        return Result.success();
    }

    /**
     * 复制-来佣比率规格明细信息
     * @param expectedSpeciesBizId 来佣比率规格明细表唯一业务ID
     * @return
     */
    @Override
    public Result copy(String expectedSpeciesBizId) {
        //校验来佣比率规格明细信息是否存在
        Result<ExpectedCommissionRatio> result = checkExpectedCommissionRatioIsExist(expectedSpeciesBizId);
        ExpectedCommissionRatio ratio = result.getData();
        ratio.setId(null);
        ratio.setExpectedCommissionRatioBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EXPECTED_COMMISSION_RATIO.getCode()));
        iExpectedCommissionRatioService.saveOrUpdate(ratio);
        return Result.success();
    }

    /**
     * 校验入参-来佣比率规格明细列表
     * @param ratioBatchSaveDtoList
     * @return
     */
    public Result checkBatchSaveRequestPram(List<ApiExpectedCommissionRatioBatchSaveDto> ratioBatchSaveDtoList) {

        // 1. 验证并准备数据
        List<ApiExpectedCommissionRatioServiceImpl.DtoWithParsedData> preparedDataList = prepareData(ratioBatchSaveDtoList);

        // 2. 检查所有数据对的年限重叠情况
        List<ApiExpectedCommissionRatioServiceImpl.OverlapError> errors = checkAllDataPairs(preparedDataList);

        // 3. 如果有错误，返回错误信息
        if (!errors.isEmpty()) {
            String errorMsg = buildErrorMessage(errors, ratioBatchSaveDtoList);
            throw new BusinessException(1004,errorMsg);
        }
        return Result.success();
    }

    /**
     * 准备数据：解析年限并转换为集合
     * @param dtoList
     * @return
     */
    private List<ApiExpectedCommissionRatioServiceImpl.DtoWithParsedData> prepareData(List<ApiExpectedCommissionRatioBatchSaveDto> dtoList) {
        List<ApiExpectedCommissionRatioServiceImpl.DtoWithParsedData> preparedList = new ArrayList<>();

        for (int i = 0; i < dtoList.size(); i++) {
            ApiExpectedCommissionRatioBatchSaveDto dto = dtoList.get(i);

            // 解析年限
            Integer startYear = ProductCommonUtils.parseYearToInt(dto.getStartPeriod());
            Integer endYear = ProductCommonUtils.parseYearToInt(dto.getEndPeriod());

            if (startYear > endYear) {
                throw new BusinessException("第" + (i+1) + "条数据的起始年限[" + startYear + "]不能大于结束年限[" + endYear + "]");
            }

            preparedList.add(new ApiExpectedCommissionRatioServiceImpl.DtoWithParsedData(i, dto, startYear, endYear));
        }

        return preparedList;
    }

    /**
     * 检查所有数据对的年限重叠情况
     */
    private List<ApiExpectedCommissionRatioServiceImpl.OverlapError> checkAllDataPairs(List<ApiExpectedCommissionRatioServiceImpl.DtoWithParsedData> preparedDataList) {
        List<ApiExpectedCommissionRatioServiceImpl.OverlapError> errors = new ArrayList<>();
        int n = preparedDataList.size();

        // 使用双重循环检查所有数据对
        for (int i = 0; i < n - 1; i++) {
            ApiExpectedCommissionRatioServiceImpl.DtoWithParsedData data1 = preparedDataList.get(i);

            for (int j = i + 1; j < n; j++) {
                ApiExpectedCommissionRatioServiceImpl.DtoWithParsedData data2 = preparedDataList.get(j);

                // 检查是否属于同一分组条件
                if (isSameGroup(data1, data2)) {
                    // 检查年限是否重叠
                    if (ProductCommonUtils.isYearRangeOverlap(data1.startYear, data1.endYear, data2.startYear, data2.endYear)) {
                        // 避免重复添加相同的错误
                        if (!isErrorAlreadyExists(errors, data1.originalIndex, data2.originalIndex)) {
                            errors.add(new ApiExpectedCommissionRatioServiceImpl.OverlapError(data1.originalIndex, data2.originalIndex));
                        }
                    }
                }
            }
        }

        return errors;
    }

    /**
     * 判断两个数据是否属于同一分组条件
     */
    private boolean isSameGroup(ApiExpectedCommissionRatioServiceImpl.DtoWithParsedData data1, ApiExpectedCommissionRatioServiceImpl.DtoWithParsedData data2) {
        // 1. 检查条件是否相同
        if (!isSameGroupExcept(data1.dto, data2.dto)) {
            return false;
        }

        return true;
    }

    /**
     * 检查条件是否相同
     */
    private boolean isSameGroupExcept(ApiExpectedCommissionRatioBatchSaveDto dto1,
                                           ApiExpectedCommissionRatioBatchSaveDto dto2) {
        return Objects.equals(dto1.getExpenseName(), dto2.getExpenseName())
                && Objects.equals(dto1.getEffectiveStart(), dto2.getEffectiveStart())
                && Objects.equals(dto1.getEffectiveEnd(), dto2.getEffectiveEnd())
                && Objects.equals(dto1.getIsExchangeRate(), dto2.getIsExchangeRate())
                && Objects.equals(dto1.getCurrency(), dto2.getCurrency());
    }

    /**
     * 检查错误是否已存在
     */
    private boolean isErrorAlreadyExists(List<ApiExpectedCommissionRatioServiceImpl.OverlapError> errors, int index1, int index2) {
        for (ApiExpectedCommissionRatioServiceImpl.OverlapError error : errors) {
            if ((error.index1 == index1 && error.index2 == index2) ||
                    (error.index1 == index2 && error.index2 == index1)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 构建错误消息
     */
    private String buildErrorMessage(List<ApiExpectedCommissionRatioServiceImpl.OverlapError> errors,
                                     List<ApiExpectedCommissionRatioBatchSaveDto> originalList) {
        if (errors.isEmpty()) {
            return "未发现重叠数据";
        }

        StringBuilder sb = new StringBuilder();

        sb.append("发现佣金年限区间重叠的数据，请检查以下数据：\n\n");
        Map<Integer, List<Integer>> overlapMap = new TreeMap<>();

        for (ApiExpectedCommissionRatioServiceImpl.OverlapError error : errors) {
            overlapMap.computeIfAbsent(error.index1, k -> new ArrayList<>()).add(error.index2);
            overlapMap.computeIfAbsent(error.index2, k -> new ArrayList<>()).add(error.index1);
        }

        // 构建错误信息
        for (Map.Entry<Integer, List<Integer>> entry : overlapMap.entrySet()) {
            int dataIndex = entry.getKey();
            List<Integer> overlapIndices = entry.getValue();

            ApiExpectedCommissionRatioBatchSaveDto dto = originalList.get(dataIndex);

            sb.append("第").append(dataIndex + 1).append("行数据年限区间")
                    .append("与以下数据年限区间有重叠:");

            for (int overlapIndex : overlapIndices) {
                ApiExpectedCommissionRatioBatchSaveDto overlapDto = originalList.get(overlapIndex);
                sb.append("  第").append(overlapIndex + 1).append("行");
            }
            sb.append("\n");
        }

        return sb.toString();
    }

    /**
     * 内部类：包含解析后数据的DTO
     */
    private static class DtoWithParsedData {
        final int originalIndex; // 在原始列表中的索引
        final ApiExpectedCommissionRatioBatchSaveDto dto; // 原始DTO
        final int startYear; // 解析后的起始年份
        final int endYear; // 解析后的结束年份

        public DtoWithParsedData(int originalIndex, ApiExpectedCommissionRatioBatchSaveDto dto,
                                 int startYear, int endYear) {
            this.originalIndex = originalIndex;
            this.dto = dto;
            this.startYear = startYear;
            this.endYear = endYear;
        }
    }

    /**
     * 内部类：重叠错误
     */
    private static class OverlapError {
        private final int index1;
        private final int index2;

        public OverlapError(int index1, int index2) {
            this.index1 = index1;
            this.index2 = index2;
        }

        public int getIndex1() { return index1; }
        public int getIndex2() { return index2; }
    }

    /**
     * 校验来佣比率规格明细信息是否存在
     * @param expectedCommissionRatioBizId
     * @return
     */
    @Override
    public Result<ExpectedCommissionRatio> checkExpectedCommissionRatioIsExist(String expectedCommissionRatioBizId) {
        ExpectedCommissionRatio expectedCommissionRatio = iExpectedCommissionRatioService.queryOne(expectedCommissionRatioBizId);
        if (Objects.isNull(expectedCommissionRatio)) {
            //数据不存在
            throw new BusinessException("来佣比率规格明细信息不存在");
        }
        return Result.success(expectedCommissionRatio);
    }
}
