package com.yd.product.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.product.api.service.ApiAnnouncementCommissionRatioService;
import com.yd.product.api.service.ApiAnnouncementSpeciesService;
import com.yd.product.api.utils.ProductCommonUtils;
import com.yd.product.feign.dto.ApiAnnouncementCommissionRatioBatchSaveDto;
import com.yd.product.feign.request.announcementcommissionratio.ApiAnnouncementCommissionRatioBatchSaveRequest;
import com.yd.product.feign.request.announcementcommissionratio.ApiAnnouncementCommissionRatioEditRequest;
import com.yd.product.feign.request.announcementcommissionratio.ApiAnnouncementCommissionRatioPageRequest;
import com.yd.product.feign.request.announcementcommissionratio.ApiAnnouncementCommissionRatioAddRequest;
import com.yd.product.feign.response.announcementcommissionratio.ApiAnnouncementCommissionRatioDetailResponse;
import com.yd.product.feign.response.announcementcommissionratio.ApiAnnouncementCommissionRatioPageResponse;
import com.yd.product.service.model.AnnouncementCommissionRatio;
import com.yd.product.service.service.IAnnouncementCommissionRatioService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiAnnouncementCommissionRatioServiceImpl implements ApiAnnouncementCommissionRatioService {

    @Autowired
    private IAnnouncementCommissionRatioService iAnnouncementCommissionRatioService;

    @Autowired
    private ApiAnnouncementSpeciesService apiAnnouncementSpeciesService;

    /**
     * 分页列表查询-公告佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiAnnouncementCommissionRatioPageResponse>> page(ApiAnnouncementCommissionRatioPageRequest request) {
        Page<ApiAnnouncementCommissionRatioPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiAnnouncementCommissionRatioPageResponse> iPage = iAnnouncementCommissionRatioService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 批量保存-公告佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result batchSave(ApiAnnouncementCommissionRatioBatchSaveRequest request) {
        Result result = checkBatchSaveRequestPram(request.getRatioBatchSaveDtoList());
        if (result.getCode() != 200) {
            return result;
        }
        List<String> announcementSpeciesBizIdList = new ArrayList<>();
        announcementSpeciesBizIdList.add(request.getAnnouncementSpeciesBizId());
        //先删后新增
        iAnnouncementCommissionRatioService.del(announcementSpeciesBizIdList);

        //新增
        if (!CollectionUtils.isEmpty(request.getRatioBatchSaveDtoList())) {
            List<AnnouncementCommissionRatio> saveList = request.getRatioBatchSaveDtoList()
                    .stream()
                    .map(dto -> {
                        AnnouncementCommissionRatio ratio = new AnnouncementCommissionRatio();
                        BeanUtils.copyProperties(dto,ratio);
                        ratio.setAnnouncementSpeciesBizId(request.getAnnouncementSpeciesBizId());
                        ratio.setAnnouncementCommissionRatioBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_ANNOUNCEMENT_COMMISSION_RATIO.getCode()));
                        return ratio;
                    }).collect(Collectors.toList());
            iAnnouncementCommissionRatioService.saveOrUpdateBatch(saveList);
        }
        return Result.success();
    }

    /**
     * 添加-公告佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result add(ApiAnnouncementCommissionRatioAddRequest request) {
        //校验公告佣比率规格信息是否存在
        apiAnnouncementSpeciesService.checkAnnouncementSpeciesIsExist(request.getAnnouncementSpeciesBizId());
        //入参校验,佣金年限校验不同条件下的唯一性 TODO
        //新增
        AnnouncementCommissionRatio ratio = new AnnouncementCommissionRatio();
        BeanUtils.copyProperties(request,ratio);
        ratio.setAnnouncementCommissionRatioBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_ANNOUNCEMENT_COMMISSION_RATIO.getCode()));
        iAnnouncementCommissionRatioService.saveOrUpdate(ratio);
        return Result.success();
    }

    /**
     * 编辑-公告佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiAnnouncementCommissionRatioEditRequest request) {
        //校验公告佣比率规格明细信息是否存在
        Result<AnnouncementCommissionRatio> result = checkAnnouncementCommissionRatioIsExist(request.getAnnouncementCommissionRatioBizId());
        AnnouncementCommissionRatio ratio = result.getData();
        //校验公告佣比率规格信息是否存在
        apiAnnouncementSpeciesService.checkAnnouncementSpeciesIsExist(request.getAnnouncementSpeciesBizId());
        //入参校验,佣金年限校验不同条件下的唯一性 TODO
        //新增
        BeanUtils.copyProperties(request,ratio);
        iAnnouncementCommissionRatioService.saveOrUpdate(ratio);
        return Result.success();
    }

    /**
     * 详情-公告佣比率规格明细信息
     * @param announcementCommissionRatioBizId 公告佣比率规格明细表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiAnnouncementCommissionRatioDetailResponse> detail(String announcementCommissionRatioBizId) {
        //校验公告佣比率规格明细信息是否存在
        Result<AnnouncementCommissionRatio> result = checkAnnouncementCommissionRatioIsExist(announcementCommissionRatioBizId);
        AnnouncementCommissionRatio announcementCommissionRatio = result.getData();
        ApiAnnouncementCommissionRatioDetailResponse response = new ApiAnnouncementCommissionRatioDetailResponse();
        BeanUtils.copyProperties(announcementCommissionRatio,response);
        return Result.success(response);
    }

    /**
     * 删除-公告佣比率规格明细信息
     * @param announcementCommissionRatioBizId 公告佣比率规格明细表唯一业务ID
     * @return
     */
    @Override
    public Result del(String announcementCommissionRatioBizId) {
        //校验公告佣比率规格明细信息是否存在
        Result<AnnouncementCommissionRatio> result = checkAnnouncementCommissionRatioIsExist(announcementCommissionRatioBizId);
        AnnouncementCommissionRatio announcementCommissionRatio = result.getData();
        iAnnouncementCommissionRatioService.removeById(announcementCommissionRatio.getId());
        return Result.success();
    }

    /**
     * 复制-公告佣比率规格明细信息
     * @param announcementCommissionRatioBizId 公告佣比率规格明细表唯一业务ID
     * @return
     */
    @Override
    public Result copy(String announcementCommissionRatioBizId) {
        //校验公告佣比率规格明细信息是否存在
        Result<AnnouncementCommissionRatio> result = checkAnnouncementCommissionRatioIsExist(announcementCommissionRatioBizId);
        AnnouncementCommissionRatio announcementCommissionRatio = result.getData();
        announcementCommissionRatio.setId(null);
        announcementCommissionRatio.setAnnouncementCommissionRatioBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_ANNOUNCEMENT_COMMISSION_RATIO.getCode()));
        iAnnouncementCommissionRatioService.saveOrUpdate(announcementCommissionRatio);
        return Result.success();
    }

    /**
     * 校验公告佣比率规格明细信息是否存在
     * @param announcementCommissionRatioBizId
     * @return
     */
    @Override
    public Result<AnnouncementCommissionRatio> checkAnnouncementCommissionRatioIsExist(String announcementCommissionRatioBizId) {
        AnnouncementCommissionRatio announcementCommissionRatio = iAnnouncementCommissionRatioService.queryOne(announcementCommissionRatioBizId);
        if (Objects.isNull(announcementCommissionRatio)) {
            //数据不存在
            throw new BusinessException("公告佣比率规格明细信息不存在");
        }
        return Result.success(announcementCommissionRatio);
    }

//    /**
//     * 校验入参-公告佣比率规格明细列表
//     * @param ratioBatchSaveDtoList
//     * @return
//     */
//    public Result checkBatchSaveRequestPram(List<ApiAnnouncementCommissionRatioBatchSaveDto> ratioBatchSaveDtoList) {
//        //校验ratioBatchSaveDtoList入参对象里面费用名称、有效开始时间、有效结束时间、适用范围、是否受汇率影响、结算币种都一样的数据情况下，判断佣金年限（起）- 佣金年限（止）区间是否有重叠情况，然后抛出异常提示哪几条数据有重叠情况。比如都一样的数据情况下，两条数据的判断佣金年限（起）- 佣金年限（止）区间一个是1-3、3-5年也算重叠，只有1-3、4-7不算重叠）
//        return null;
//    }

    /**
     * 校验入参-公告佣比率规格明细列表
     * @param ratioBatchSaveDtoList
     * @return
     */
    public Result checkBatchSaveRequestPram(List<ApiAnnouncementCommissionRatioBatchSaveDto> ratioBatchSaveDtoList) {
        if (ratioBatchSaveDtoList == null || ratioBatchSaveDtoList.isEmpty()) {
            return Result.success("数据为空，无需校验");
        }

        try {
            // 1. 验证并准备数据
            List<DtoWithParsedData> preparedDataList = prepareData(ratioBatchSaveDtoList);
            if (preparedDataList == null) {
                return Result.fail("数据准备失败，请检查输入数据");
            }

            // 2. 检查所有数据对的年限重叠情况（考虑scope交集）
            List<OverlapError> errors = checkAllDataPairs(preparedDataList);

            // 3. 如果有错误，返回错误信息
            if (!errors.isEmpty()) {
                String errorMsg = buildErrorMessage(errors, ratioBatchSaveDtoList);
                return Result.fail(errorMsg);
            }

            return Result.success("校验通过，无重叠数据");
        } catch (Exception e) {
            return Result.fail("数据校验异常：" + e.getMessage());
        }
    }

    /**
     * 准备数据：解析年限并将scope转换为集合
     * @param dtoList
     * @return
     */
    private List<DtoWithParsedData> prepareData(List<ApiAnnouncementCommissionRatioBatchSaveDto> dtoList) {
        List<DtoWithParsedData> preparedList = new ArrayList<>();

        for (int i = 0; i < dtoList.size(); i++) {
            ApiAnnouncementCommissionRatioBatchSaveDto dto = dtoList.get(i);

            // 解析年限
            Integer startYear = ProductCommonUtils.parseYearToInt(dto.getStartPeriod());
            Integer endYear = ProductCommonUtils.parseYearToInt(dto.getEndPeriod());

            if (startYear > endYear) {
                throw new BusinessException("第" + (i+1) + "条数据的起始年限[" + startYear + "]不能大于结束年限[" + endYear + "]");
            }

            // 解析scope为集合,去重
            Set<String> scopeSet = ProductCommonUtils.parseScopeToSet(dto.getScope());

            preparedList.add(new DtoWithParsedData(i, dto, startYear, endYear, scopeSet));
        }

        return preparedList;
    }

    /**
     * 检查所有数据对的年限重叠情况
     */
    private List<OverlapError> checkAllDataPairs(List<DtoWithParsedData> preparedDataList) {
        List<OverlapError> errors = new ArrayList<>();
        int n = preparedDataList.size();

        // 使用双重循环检查所有数据对
        for (int i = 0; i < n - 1; i++) {
            DtoWithParsedData data1 = preparedDataList.get(i);

            for (int j = i + 1; j < n; j++) {
                DtoWithParsedData data2 = preparedDataList.get(j);

                // 检查是否属于同一分组条件（考虑scope交集）
                if (isSameGroup(data1, data2)) {
                    // 检查年限是否重叠
                    if (ProductCommonUtils.isYearRangeOverlap(data1.startYear, data1.endYear, data2.startYear, data2.endYear)) {
                        // 避免重复添加相同的错误
                        if (!isErrorAlreadyExists(errors, data1.originalIndex, data2.originalIndex)) {
                            errors.add(new OverlapError(data1.originalIndex, data2.originalIndex));
                        }
                    }
                }
            }
        }

        return errors;
    }

    /**
     * 判断两个数据是否属于同一分组条件
     */
    private boolean isSameGroup(DtoWithParsedData data1, DtoWithParsedData data2) {
        // 1. 检查除scope外的其他条件是否相同
        if (!isSameGroupExceptScope(data1.dto, data2.dto)) {
            return false;
        }

        // 2. 检查scope是否有交集
        return ProductCommonUtils.hasScopeIntersection(data1.scopeSet, data2.scopeSet);
    }

    /**
     * 检查除scope外的其他条件是否相同
     */
    private boolean isSameGroupExceptScope(ApiAnnouncementCommissionRatioBatchSaveDto dto1,
                                           ApiAnnouncementCommissionRatioBatchSaveDto dto2) {
        return Objects.equals(dto1.getExpenseName(), dto2.getExpenseName())
                && Objects.equals(dto1.getEffectiveStart(), dto2.getEffectiveStart())
                && Objects.equals(dto1.getEffectiveEnd(), dto2.getEffectiveEnd())
                && Objects.equals(dto1.getIsExchangeRate(), dto2.getIsExchangeRate())
                && Objects.equals(dto1.getCurrency(), dto2.getCurrency());
    }

    /**
     * 校验基础字段
     */
    private boolean validateBasicFields(ApiAnnouncementCommissionRatioBatchSaveDto dto) {
        return dto.getExpenseName() != null && !dto.getExpenseName().trim().isEmpty()
                && dto.getEffectiveStart() != null
                && dto.getEffectiveEnd() != null
                && dto.getEffectiveStart().isBefore(dto.getEffectiveEnd())
                && dto.getScope() != null && !dto.getScope().trim().isEmpty()
                && dto.getIsExchangeRate() != null && !dto.getIsExchangeRate().trim().isEmpty()
                && dto.getCurrency() != null && !dto.getCurrency().trim().isEmpty();
    }

    /**
     * 检查错误是否已存在
     */
    private boolean isErrorAlreadyExists(List<OverlapError> errors, int index1, int index2) {
        for (OverlapError error : errors) {
            if ((error.index1 == index1 && error.index2 == index2) ||
                    (error.index1 == index2 && error.index2 == index1)) {
                return true;
            }
        }
        return false;
    }

//    /**
//     * 构建错误消息
//     */
//    private String buildErrorMessage(List<OverlapError> errors,
//                                     List<ApiAnnouncementCommissionRatioBatchSaveDto> originalList) {
//        if (errors.isEmpty()) {
//            return "未发现重叠数据";
//        }
//
//        StringBuilder sb = new StringBuilder();
//
//        Set<Integer> set = new HashSet<>();
//        if (!CollectionUtils.isEmpty(errors)) {
//            errors.stream().forEach(dto -> {
//                set.add(dto.getIndex1());
//                set.add(dto.getIndex2());
//            });
//        }
//
//        sb.append("发现佣金年限区间重叠的数据，请检查以下行数：\n\n");
//
//        // 将行数（索引+1，因为用户从1开始计数）排序并格式化为"第X行"
//        String rowNumbers = set.stream()
//                .sorted()
//                .map(index -> "第" + (index + 1) + "行")
//                .collect(Collectors.joining("，")); // 用中文逗号分隔
//
//        sb.append(rowNumbers);
//
//        return sb.toString();
//    }

    /**
     * 构建错误消息
     */
    private String buildErrorMessage(List<OverlapError> errors,
                                     List<ApiAnnouncementCommissionRatioBatchSaveDto> originalList) {
        if (errors.isEmpty()) {
            return "未发现重叠数据";
        }

        StringBuilder sb = new StringBuilder();

//        Set<Integer> set = new HashSet<>();
//        if (!CollectionUtils.isEmpty(errors)) {
//            errors.stream().forEach(dto -> {
//                set.add(dto.getIndex1());
//                set.add(dto.getIndex2());
//            });
//        }
        sb.append("发现佣金年限区间重叠的数据，请检查以下数据：\n\n");
//        sb.append("发现佣金年限区间重叠的数据，请检查以下行数：\n\n");
//        sb.append(String.join("",set))
//        // 按数据索引分组，方便查看每条数据的所有重叠关系
        Map<Integer, List<Integer>> overlapMap = new TreeMap<>();

        for (OverlapError error : errors) {
            overlapMap.computeIfAbsent(error.index1, k -> new ArrayList<>()).add(error.index2);
            overlapMap.computeIfAbsent(error.index2, k -> new ArrayList<>()).add(error.index1);
        }

        // 构建错误信息
        for (Map.Entry<Integer, List<Integer>> entry : overlapMap.entrySet()) {
            int dataIndex = entry.getKey();
            List<Integer> overlapIndices = entry.getValue();

            ApiAnnouncementCommissionRatioBatchSaveDto dto = originalList.get(dataIndex);

            sb.append("第").append(dataIndex + 1).append("行数据年限区间")
//                    .append("  - 费用名称: ").append(dto.getExpenseName()).append("\n")
//                    .append("  - 年限区间: ").append(dto.getStartPeriod()).append(" 至 ").append(dto.getEndPeriod()).append("\n")
//                    .append("  - 适用范围: ").append(dto.getScope()).append("\n")
//                    .append("  - 有效时间: ").append(dto.getEffectiveStart()).append(" 至 ").append(dto.getEffectiveEnd()).append("\n")
//                    .append("  - 汇率影响: ").append(dto.getIsExchangeRate()).append("\n")
//                    .append("  - 结算币种: ").append(dto.getCurrency()).append("\n")
                    .append("与以下数据年限区间有重叠:");

            for (int overlapIndex : overlapIndices) {
                ApiAnnouncementCommissionRatioBatchSaveDto overlapDto = originalList.get(overlapIndex);
                sb.append("  第").append(overlapIndex + 1).append("行");
//                        .append(overlapDto.getExpenseName())
//                        .append(" [").append(overlapDto.getStartPeriod())
//                        .append("-").append(overlapDto.getEndPeriod()).append("]")
//                        .append(" 适用范围:").append(overlapDto.getScope()).append("\n");
            }
            sb.append("\n");
        }

        // 添加通用分组条件说明
//        sb.append("重叠判断规则说明:\n");
//        sb.append("1. 以下条件完全相同的数据才会被比较:\n");
//        sb.append("   - 费用名称\n");
//        sb.append("   - 有效开始时间\n");
//        sb.append("   - 有效结束时间\n");
//        sb.append("   - 是否受汇率影响\n");
//        sb.append("   - 结算币种\n");
//        sb.append("2. 适用范围只要有一个相同的值（交集）就视为相同分组条件\n");
//        sb.append("3. 年限区间重叠判断规则:\n");
//        sb.append("   - 1-3 和 3-5: 算重叠（因为有共同的3）\n");
//        sb.append("   - 1-3 和 4-7: 不算重叠\n");
//        sb.append("   - 1-5 和 2-4: 算重叠\n");

        return sb.toString();
    }

    /**
     * 内部类：包含解析后数据的DTO
     */
    private static class DtoWithParsedData {
        final int originalIndex; // 在原始列表中的索引
        final ApiAnnouncementCommissionRatioBatchSaveDto dto; // 原始DTO
        final int startYear; // 解析后的起始年份
        final int endYear; // 解析后的结束年份
        final Set<String> scopeSet; // 解析后的scope集合

        public DtoWithParsedData(int originalIndex, ApiAnnouncementCommissionRatioBatchSaveDto dto,
                                 int startYear, int endYear, Set<String> scopeSet) {
            this.originalIndex = originalIndex;
            this.dto = dto;
            this.startYear = startYear;
            this.endYear = endYear;
            this.scopeSet = scopeSet;
        }
    }

    /**
     * 内部类：重叠错误
     */
    private static class OverlapError {
        private final int index1;
        private final int index2;

        public OverlapError(int index1, int index2) {
            this.index1 = index1;
            this.index2 = index2;
        }

        public int getIndex1() { return index1; }
        public int getIndex2() { return index2; }
    }

}
