package com.yd.product.api.service.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.product.api.service.ApiAnnouncementCommissionRatioService;
import com.yd.product.api.service.ApiProductLaunchService;
import com.yd.product.feign.dto.ApiSpeciesSettingDto;
import com.yd.product.feign.request.announcementcommissionratio.ApiAnnouncementCommissionRatioPageRequest;
import com.yd.product.feign.request.announcementcommissionratio.ApiAnnouncementCommissionRatioSaveRequest;
import com.yd.product.feign.response.announcementcommissionratio.ApiAnnouncementCommissionRatioDetailResponse;
import com.yd.product.feign.response.announcementcommissionratio.ApiAnnouncementCommissionRatioPageResponse;
import com.yd.product.service.model.AnnouncementCommissionRatio;
import com.yd.product.service.model.ProductLaunch;
import com.yd.product.service.service.IAnnouncementCommissionRatioService;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiAnnouncementCommissionRatioServiceImpl implements ApiAnnouncementCommissionRatioService {

    @Autowired
    private IAnnouncementCommissionRatioService iAnnouncementCommissionRatioService;

    @Autowired
    private ApiProductLaunchService apiProductLaunchService;

    /**
     * 分页列表查询-公告佣比率信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiAnnouncementCommissionRatioPageResponse>> page(ApiAnnouncementCommissionRatioPageRequest request) {
        Page<ApiAnnouncementCommissionRatioPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiAnnouncementCommissionRatioPageResponse> iPage = iAnnouncementCommissionRatioService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 保存-公告佣比率信息
     * @param request
     * @return
     */
    @Override
    public Result save(ApiAnnouncementCommissionRatioSaveRequest request) {
        //校验产品上架信息是否存在
        Result<ProductLaunch> result = apiProductLaunchService.checkProductLaunchIsExist(request.getProductLaunchBizId());
        ProductLaunch productLaunch = result.getData();
        //先删后新增
        //删除
        iAnnouncementCommissionRatioService.delByProductLaunchBizId(request.getProductLaunchBizId());

        //新增
        List<AnnouncementCommissionRatio> expectedCommissionRatioList = request.getApiAnnouncementCommissionRatioDtoList()
                .stream().map(dto -> {
                    AnnouncementCommissionRatio ratio = new AnnouncementCommissionRatio();
                    BeanUtils.copyProperties(dto,ratio);
                    ratio.setSpeciesJson(!CollectionUtils.isEmpty(dto.getApiSpeciesSettingDtoList()) ? JSON.toJSONString(dto.getApiSpeciesSettingDtoList()) : "");
                    return ratio;
                }).collect(Collectors.toList());
        iAnnouncementCommissionRatioService.saveBatch(expectedCommissionRatioList);
        return Result.success();
    }

    /**
     * 详情-公告佣比率信息
     * @param announcementCommissionRatioBizId 公告佣比率表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiAnnouncementCommissionRatioDetailResponse> detail(String announcementCommissionRatioBizId) {
        //校验公告佣比率信息是否存在
        Result<AnnouncementCommissionRatio> result = checkAnnouncementCommissionRatioIsExist(announcementCommissionRatioBizId);
        AnnouncementCommissionRatio announcementCommissionRatio = result.getData();
        ApiAnnouncementCommissionRatioDetailResponse response = new ApiAnnouncementCommissionRatioDetailResponse();
        BeanUtils.copyProperties(announcementCommissionRatio,response);
        response.setApiSpeciesSettingDtoList(StringUtil.isNotBlank(announcementCommissionRatio.getSpeciesJson()) ? JSON.parseArray(announcementCommissionRatio.getSpeciesJson(), ApiSpeciesSettingDto.class) : new ArrayList<>());
        return Result.success(response);
    }

    /**
     * 校验公告佣比率信息是否存在
     * @param announcementCommissionRatioBizId
     * @return
     */
    @Override
    public Result<AnnouncementCommissionRatio> checkAnnouncementCommissionRatioIsExist(String announcementCommissionRatioBizId) {
        AnnouncementCommissionRatio announcementCommissionRatio = iAnnouncementCommissionRatioService.queryOne(announcementCommissionRatioBizId);
        if (Objects.isNull(announcementCommissionRatio)) {
            //数据不存在
            throw new BusinessException("公告佣比率信息不存在");
        }
        return Result.success(announcementCommissionRatio);
    }

}
