package com.yd.product.api.service.impl;

import com.yd.base.feign.client.fieldvalue.ApiFieldValueFeignClient;
import com.yd.base.feign.request.fieldvalue.ApiFieldValueListRequest;
import com.yd.base.feign.response.fieldvalue.ApiFieldValueDetailResponse;
import com.yd.common.enums.CommonEnum;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.product.api.service.ApiAttributeSettingService;
import com.yd.product.feign.response.productlaunch.ApiAttributeSettingDto;
import com.yd.product.service.dto.AttributeSettingDto;
import com.yd.product.service.model.AttributeSetting;
import com.yd.product.service.service.IAttributeSettingService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiAttributeSettingServiceImpl implements ApiAttributeSettingService {

    @Autowired
    private IAttributeSettingService iAttributeSettingService;

    @Autowired
    private ApiFieldValueFeignClient apiFieldValueFeignClient;

    /**
     * 保存产品属性（参数）配置信息
     * @param apiAttributeSettingDtoList
     * @param productLaunchBizId
     * @return
     */
    @Override
    public Result save(List<ApiAttributeSettingDto> apiAttributeSettingDtoList, String productLaunchBizId) {
        //先删后新增
        iAttributeSettingService.delByProductLaunchBizId(productLaunchBizId);

        //新增
        if (!CollectionUtils.isEmpty(apiAttributeSettingDtoList)) {
            List<AttributeSetting> attributeSettingList = apiAttributeSettingDtoList.stream().map(dto -> {
                AttributeSetting attributeSetting = new AttributeSetting();
                BeanUtils.copyProperties(dto,attributeSetting);
                attributeSetting.setProductLaunchBizId(productLaunchBizId);
                attributeSetting.setAttributeSettingBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_ATTRIBUTE_SETTING.getCode()));
                return attributeSetting;
            }).collect(Collectors.toList());
            iAttributeSettingService.saveOrUpdateBatch(attributeSettingList);
        }
        return Result.success();
    }

    /**
     * 根据产品上架信息id查询产品属性（参数）配置列表
     * @param productLaunchBizId
     * @return
     */
    @Override
    public List<ApiAttributeSettingDto> queryAttributeSettingDtoList(String productLaunchBizId) {
        List<ApiAttributeSettingDto> apiAttributeSettingDtoList = new ArrayList<>();
        List<AttributeSetting> attributeSettingList = iAttributeSettingService.queryList(AttributeSettingDto.builder()
                .productLaunchBizId(productLaunchBizId)
                .build());

        if (!CollectionUtils.isEmpty(attributeSettingList)) {
            // 存储每个属性设置的原始值和可能的业务ID
            Map<String, List<String>> settingDtoToBizIdsMap = new HashMap<>();
            Map<String, List<String>> settingDtoToOriginalValuesMap = new HashMap<>();
            List<String> allFieldValueBizIds = new ArrayList<>();

            for (int i = 0; i < attributeSettingList.size(); i++) {
                AttributeSetting dto = attributeSettingList.get(i);
                List<String> bizIds = new ArrayList<>();
                List<String> originalValues = new ArrayList<>();

                // 处理原始值
                if (StringUtils.isNotBlank(dto.getValue())) {
                    if (dto.getValue().contains(";")) {
                        // 多选情况
                        originalValues.addAll(Arrays.asList(dto.getValue().split(";")));
                    } else {
                        // 单选情况
                        originalValues.add(dto.getValue());
                    }
                }

                // 收集业务ID
                // 1. 首先从fieldValueBizId字段收集
                if (StringUtils.isNotBlank(dto.getFieldValueBizId())) {
                    if (dto.getFieldValueBizId().contains(";")) {
                        bizIds.addAll(Arrays.asList(dto.getFieldValueBizId().split(";")));
                    } else {
                        bizIds.add(dto.getFieldValueBizId());
                    }
                }
                // 2. 如果fieldValueBizId为空，但originalValues看起来像是业务ID，也收集
                else if (!originalValues.isEmpty()) {
                    // 检查所有原始值是否都像是业务ID
                    boolean allValuesLookLikeBizIds = originalValues.stream()
                            .allMatch(this::isLikelyBizId);

                    if (allValuesLookLikeBizIds) {
                        bizIds.addAll(originalValues);
                    }
                }

                settingDtoToBizIdsMap.put(String.valueOf(i), bizIds);
                settingDtoToOriginalValuesMap.put(String.valueOf(i), originalValues);

                if (!bizIds.isEmpty()) {
                    allFieldValueBizIds.addAll(bizIds);
                }
            }

            //批量查询字段值详情
            Map<String, String> fieldValueBizIdToNameMap = new HashMap<>();
            Map<String, String> fieldValueBizIdToCodeMap = new HashMap<>();
            if (!CollectionUtils.isEmpty(allFieldValueBizIds)) {
                // 去重
                List<String> distinctFieldValueBizIds = allFieldValueBizIds.stream()
                        .distinct()
                        .collect(Collectors.toList());

                ApiFieldValueListRequest request = new ApiFieldValueListRequest();
                request.setFieldValueBizIdList(distinctFieldValueBizIds);

                try {
                    Result<List<ApiFieldValueDetailResponse>> result = apiFieldValueFeignClient.list(request);
                    if (result != null && result.getCode() == 200 && !CollectionUtils.isEmpty(result.getData())) {
                        log.info("查询到字段值详情数量: {}", result.getData().size());

                        // 构建映射关系：字段值业务ID -> 字段值名称 和 字段值业务ID -> 字段值编码
                        for (ApiFieldValueDetailResponse response : result.getData()) {
                            log.debug("字段值详情 - fieldValueBizId: {}, fieldValueCode: {}, value: {}",
                                    response.getFieldValueBizId(), response.getFieldValueCode(), response.getValue());

                            fieldValueBizIdToNameMap.put(response.getFieldValueBizId(), response.getValue());

                            // 如果fieldValueCode不为空，则放入映射表
                            if (StringUtils.isNotBlank(response.getFieldValueCode())) {
                                fieldValueBizIdToCodeMap.put(response.getFieldValueBizId(), response.getFieldValueCode());
                            }
                        }

                        log.info("字段值编码映射表: {}", fieldValueBizIdToCodeMap);
                    } else {
                        log.warn("查询字段值详情失败，fieldValueBizIdList: {}", distinctFieldValueBizIds);
                    }
                } catch (Exception e) {
                    log.error("调用字段值服务异常", e);
                }
            }

            //构建返回结果
            for (int i = 0; i < attributeSettingList.size(); i++) {
                AttributeSetting dto = attributeSettingList.get(i);
                ApiAttributeSettingDto settingDto = new ApiAttributeSettingDto();
                BeanUtils.copyProperties(dto, settingDto);

                List<String> bizIds = settingDtoToBizIdsMap.get(String.valueOf(i));
                List<String> originalValues = settingDtoToOriginalValuesMap.get(String.valueOf(i));

                if (!bizIds.isEmpty()) {
                    // 有业务ID的情况
                    Map<String, String> finalFieldValueBizIdToNameMap = fieldValueBizIdToNameMap;
                    Map<String, String> finalFieldValueBizIdToCodeMap = fieldValueBizIdToCodeMap;

                    List<String> valueNames = bizIds.stream()
                            .map(bizId -> {
                                // 首先尝试从字段值服务查询名称
                                String name = finalFieldValueBizIdToNameMap.get(bizId);
                                if (StringUtils.isNotBlank(name)) {
                                    return name;
                                }
                                // 如果查询不到，使用原始值（如果原始值不是业务ID）或业务ID本身
                                // 这里需要找到对应的原始值
                                int index = bizIds.indexOf(bizId);
                                if (index >= 0 && index < originalValues.size()) {
                                    String originalValue = originalValues.get(index);
                                    // 如果原始值看起来像是业务ID，则使用业务ID本身
                                    if (isLikelyBizId(originalValue)) {
                                        return bizId;
                                    }
                                    return originalValue;
                                }
                                return bizId;
                            })
                            .collect(Collectors.toList());
                    settingDto.setValueNames(StringUtils.join(valueNames, ";"));

                    List<String> valueCodes = bizIds.stream()
                            .map(bizId -> {
                                String code = finalFieldValueBizIdToCodeMap.get(bizId);
                                return StringUtils.isNotBlank(code) ? code : "";
                            })
                            .collect(Collectors.toList());
                    settingDto.setValueCodes(StringUtils.join(valueCodes, ";"));
                } else if (!originalValues.isEmpty()) {
                    // 没有业务ID，但有原始值的情况
                    settingDto.setValueNames(StringUtils.join(originalValues, ";"));
                    settingDto.setValueCodes("");
                }

                apiAttributeSettingDtoList.add(settingDto);
            }
        }
        return apiAttributeSettingDtoList;
    }

    /**
     * 判断一个字符串是否可能是业务ID
     */
    private boolean isLikelyBizId(String str) {
        if (StringUtils.isBlank(str)) {
            return false;
        }

        // 业务ID通常包含下划线，且长度较长
        // 这是一个保守的判断
        return str.contains("_") && str.length() > 10;
    }
}