package com.yd.question.api.service.impl;

import com.alibaba.fastjson.JSON;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.question.api.service.ApiQuestionnairesService;
import com.yd.question.feign.dto.*;
import com.yd.question.feign.request.ApiAnswerSaveRequest;
import com.yd.question.feign.request.ApiObjectSaveRequest;
import com.yd.question.feign.response.ApiAnswerSaveResponse;
import com.yd.question.feign.response.ApiQuestionnairesDetailResponse;
import com.yd.question.service.dto.AnswerSessionsDto;
import com.yd.question.service.dto.RelObjectQuestionnaireDto;
import com.yd.question.service.model.*;
import com.yd.question.service.service.*;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiQuestionnairesServiceImpl implements ApiQuestionnairesService {

    @Autowired
    private IQuestionnairesService iQuestionnairesService;

    @Autowired
    private IRelObjectQuestionnaireService iRelObjectQuestionnaireService;

    @Autowired
    private IOptionsService iOptionsService;

    @Autowired
    private IRelQuestionsOptionsService iRelQuestionsOptionsService;

    @Autowired
    private IRelOptionsAddQuestionsService iRelOptionsAddQuestionsService;

    @Autowired
    private IAddQuestionsService iAddQuestionsService;

    @Autowired
    private IAnswerSessionsService iAnswerSessionsService;


    /**
     * 获取问卷详细信息-单个问卷详细信息
     * @param questionnaireBizId 问卷唯一业务ID
     * @param objectBizId 对象业务ID
     * @return
     */
    @Override
    public Result<ApiQuestionnairesDetailResponse> detail(String questionnaireBizId, String objectBizId) {
        List<RelObjectQuestionnaire> list = iRelObjectQuestionnaireService.queryList(RelObjectQuestionnaireDto.builder()
                .objectBizId(objectBizId)
                .build());
        if (CollectionUtils.isEmpty(list)) {
            throw new BusinessException("对象问卷关系不存在");
        }
        RelObjectQuestionnaire relObjectQuestionnaire = list.get(0);
        questionnaireBizId = relObjectQuestionnaire.getQuestionnaireBizId();

        //校验问卷信息是否存在
        Result<Questionnaires> result = checkQuestionnairesIsExist(questionnaireBizId);
        Questionnaires questionnaires = result.getData();

        ApiQuestionnairesDetailResponse response = new ApiQuestionnairesDetailResponse();

        //问卷基础信息
        ApiQuestionnairesDto questionnairesDto = new ApiQuestionnairesDto();
        BeanUtils.copyProperties(questionnaires,questionnairesDto);
        response.setQuestionnairesDto(questionnairesDto);

        //一二级分类分组的问题列表信息
        //根据问卷业务id查询：问卷问题关联表、分组问题关联表、问题表。三张表关联数据
        List<ApiRelQuestionsGroupDto> relList = iQuestionnairesService.relList(questionnaireBizId);

        //根据问卷绑定的问题列表的分类情况展示对应不同的分层结构
        response = selectList(response,relList,questionnaireBizId,objectBizId);

        return Result.success(response);
    }

    /**
     * 根据问卷绑定的问题列表的分类情况展示对应不同的分层结构
     * @param relList
     * @param questionnaireBizId
     * @param objectBizId
     * @return
     */
    public ApiQuestionnairesDetailResponse selectList(
                                                      ApiQuestionnairesDetailResponse response,
                                                      List<ApiRelQuestionsGroupDto> relList,
                                                      String questionnaireBizId,
                                                      String objectBizId) {
        if (!CollectionUtils.isEmpty(relList)){
            //第一级分类并且第二级分类都不能为空的列表size必须和relList一样
            List<ApiRelQuestionsGroupDto> filterList = relList.stream().filter(d -> StringUtil.isNotBlank(d.getFirstCategory())
                    && StringUtil.isNotBlank(d.getSecondCategory())).collect(Collectors.toList());
            if (filterList.size() == relList.size()) {
                //一二级分类分组的问题列表信息
                List<ApiFirstAndSecondCategoryDto> firstAndSecondCategoryDtoList = getApiFirstAndSecondCategoryDtoList(relList,questionnaireBizId,objectBizId);
                response.setFirstAndSecondCategoryDtoList(firstAndSecondCategoryDtoList);
                return response;
            }

            //第一级分类不为空并且第二级分类为空的列表size必须和relList一样
            List<ApiRelQuestionsGroupDto> filterList1 = relList.stream().filter(d -> StringUtil.isNotBlank(d.getFirstCategory())
                    && StringUtil.isBlank(d.getSecondCategory())).collect(Collectors.toList());
            if (filterList1.size() == relList.size()) {
                //一级分类分组的问题列表信息
                List<ApiFirstCategoryDto> firstCategoryDtoList = getFirstCategoryDtoList(relList,questionnaireBizId,objectBizId);
                response.setFirstCategoryDtoList(firstCategoryDtoList);
                return response;
            }

            //第一级分类并且第二级分类都为空的列表size必须和relList一样
            List<ApiRelQuestionsGroupDto> filterList2 = relList.stream().filter(d -> StringUtil.isNotBlank(d.getFirstCategory())
                    && StringUtil.isBlank(d.getSecondCategory())).collect(Collectors.toList());
            if (filterList2.size() == relList.size()) {
                //一级分类分组的问题列表信息
                List<ApiQuestionsDto> apiQuestionsDtoList = questionsDtoList(relList,questionnaireBizId,objectBizId);
                response.setQuestionsDtoList(apiQuestionsDtoList);
                return response;
            }
        }
        return response;
    }

    /**
     * 一级分类分组的问题列表信息
     * @param relList
     * @param questionnaireBizId
     * @param objectBizId
     * @return
     */
    public List<ApiFirstCategoryDto> getFirstCategoryDtoList(List<ApiRelQuestionsGroupDto> relList,
                                                             String questionnaireBizId,
                                                             String objectBizId) {
        //进行多级分组并转换为目标结构
        List<ApiFirstCategoryDto> firstCategoryDtoList = relList.stream()
                // 过滤掉firstCategory为null的记录
                .filter(dto -> dto.getFirstCategory() != null)
                // 按一级分类分组
                .collect(Collectors.groupingBy(
                        ApiRelQuestionsGroupDto::getFirstCategory,
                        Collectors.toList()
                ))
                .entrySet().stream()
                .map(entry -> {
                    // 创建一级分类DTO
                    ApiFirstCategoryDto firstCategoryDto = new ApiFirstCategoryDto();
                    firstCategoryDto.setFirstCategory(entry.getKey());

                    // 设置问题列表
                    List<ApiRelQuestionsGroupDto> questionsDtoList = entry.getValue();
                    if (!CollectionUtils.isEmpty(questionsDtoList)) {

                        //设置问题列表
                        firstCategoryDto.setQuestionsDtoList(questionsDtoList(questionsDtoList,questionnaireBizId,objectBizId));
                    }

                    return firstCategoryDto;
                })
                .collect(Collectors.toList());

        return firstCategoryDtoList;
    }

    /**
     * 一二级分类分组的问题列表信息
     * @param relList 根据问卷业务id查询：问卷问题关联表、分组问题关联表、问题表。三张表关联数据
     * @param questionnaireBizId
     * @param objectBizId
     * @return
     */
    public List<ApiFirstAndSecondCategoryDto> getApiFirstAndSecondCategoryDtoList(List<ApiRelQuestionsGroupDto> relList,
                                                                                  String questionnaireBizId,
                                                                                  String objectBizId) {

        //进行多级分组并转换为目标结构
        List<ApiFirstAndSecondCategoryDto> firstAndSecondCategoryDtoList = relList.stream()
                // 过滤掉firstCategory为null的记录
                .filter(dto -> dto.getFirstCategory() != null)
                // 按一级分类分组
                .collect(Collectors.groupingBy(
                        ApiRelQuestionsGroupDto::getFirstCategory,
                        Collectors.toList()
                ))
                .entrySet().stream()
                .map(entry -> {
                    // 创建一级分类DTO
                    ApiFirstAndSecondCategoryDto firstCategoryDto = new ApiFirstAndSecondCategoryDto();
                    firstCategoryDto.setFirstCategory(entry.getKey());

                    // 对当前一级分类下的数据进行二级分类分组
                    Map<String, List<ApiRelQuestionsGroupDto>> secondCategoryMap = entry.getValue().stream()
                            // 过滤掉secondCategory为null的记录
                            .filter(dto -> dto.getSecondCategory() != null)
                            .collect(Collectors.groupingBy(ApiRelQuestionsGroupDto::getSecondCategory));

                    // 转换为二级分类DTO列表
                    List<ApiSecondCategoryDto> secondCategoryDtoList = secondCategoryMap.entrySet().stream()
                            .map(secondEntry -> {
                                ApiSecondCategoryDto secondCategoryDto = new ApiSecondCategoryDto();
                                secondCategoryDto.setSecondCategory(secondEntry.getKey());

                                // 设置问题列表
                                List<ApiRelQuestionsGroupDto> questionsDtoList = secondEntry.getValue();
                                if (!CollectionUtils.isEmpty(questionsDtoList)) {

                                    //设置问题列表
                                    secondCategoryDto.setQuestionsDtoList(questionsDtoList(questionsDtoList,questionnaireBizId,objectBizId));
                                }
                                return secondCategoryDto;
                            })
                            .collect(Collectors.toList());

                    // 设置二级分类列表到一级分类DTO
                    firstCategoryDto.setSecondCategoryDtoList(secondCategoryDtoList);

                    return firstCategoryDto;
                })
                .collect(Collectors.toList());

        return firstAndSecondCategoryDtoList;
    }

    /**
     * 获取问题列表
     * @param questionsDtoList
     * @param questionnaireBizId
     * @param objectBizId
     * @return
     */
    public List<ApiQuestionsDto> questionsDtoList(List<ApiRelQuestionsGroupDto> questionsDtoList,
                                                  String questionnaireBizId,
                                                  String objectBizId) {
        List<ApiQuestionsDto> apiQuestionsDtoList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(questionsDtoList)) {
            apiQuestionsDtoList = questionsDtoList.stream().map(d -> {
                ApiQuestionsDto apiQuestionsDto = new ApiQuestionsDto();
                BeanUtils.copyProperties(d,apiQuestionsDto);
                apiQuestionsDto.setId(d.getQuestionId());
                //问题的选项列表信息
                List<ApiOptionsDto> apiOptionsDtoList = optionsDtoList(questionnaireBizId,d.getQuestionBizId());
                apiQuestionsDto.setOptionsDtoList(apiOptionsDtoList);
                if (!CollectionUtils.isEmpty(apiOptionsDtoList)) {
                    List<String> optionBizIdList = apiOptionsDtoList.stream().map(ApiOptionsDto::getOptionBizId).collect(Collectors.toList());
                    apiQuestionsDto.setAddQuestionsDtoList(addQuestionsDtoList(questionnaireBizId,d.getQuestionBizId(),optionBizIdList));
                }
                //答题会话对象（实际回答的内容）
                apiQuestionsDto.setAnswerSessionsDto(answerSessionsDto(objectBizId,questionnaireBizId,d.getQuestionBizId()));

                return apiQuestionsDto;
            }).collect(Collectors.toList());
        }
        return apiQuestionsDtoList;
    }

    /**
     * 获取问题的选项列表信息
     * @param questionnaireBizId 问卷唯一业务ID
     * @param questionBizId 问题唯一业务ID
     * @return
     */
    public List<ApiOptionsDto> optionsDtoList(String questionnaireBizId,String questionBizId) {
        List<ApiOptionsDto> apiOptionsDtoList = new ArrayList<>();
        //查询问题选项关系
        List<RelQuestionsOptions> relQuestionsOptionsList = iRelQuestionsOptionsService.queryList(questionnaireBizId,questionBizId);
        if (!CollectionUtils.isEmpty(relQuestionsOptionsList)) {
            //选项id列表
            List<String> optionBizIdList = relQuestionsOptionsList
                    .stream()
                    .map(RelQuestionsOptions::getOptionBizId)
                    .collect(Collectors.toList());

            //根据选项id列表查询选项对象列表
            List<Options> optionsList = iOptionsService.queryList("",optionBizIdList);
            if (!CollectionUtils.isEmpty(optionsList)) {
                apiOptionsDtoList = optionsList.stream().map(d -> {
                    ApiOptionsDto apiOptionsDto = new ApiOptionsDto();
                    BeanUtils.copyProperties(d,apiOptionsDto);
                    //选项附加问题列表
//                    apiOptionsDto.setAddQuestionsDtoList(addQuestionsDtoList(questionnaireBizId,questionBizId,d.getOptionBizId()));
                    return apiOptionsDto;
                }).collect(Collectors.toList());
            }
        }

        return apiOptionsDtoList;
    }

    /**
     * 选项附加问题列表
     * @param questionnaireBizId
     * @param questionBizId
     * @param optionBizIdList
     * @return
     */
    public List<ApiAddQuestionsDto> addQuestionsDtoList(String questionnaireBizId,
                                                        String questionBizId,
                                                        List<String> optionBizIdList) {
        List<ApiAddQuestionsDto> apiAddQuestionsDtoList = new ArrayList<>();
        List<RelOptionsAddQuestions> relOptionsAddQuestionsList = iRelOptionsAddQuestionsService.queryList(questionnaireBizId,questionBizId,optionBizIdList);
        if (!CollectionUtils.isEmpty(relOptionsAddQuestionsList)) {
            List<String> addQuestionsBizIdList = relOptionsAddQuestionsList.stream()
                    .map(RelOptionsAddQuestions::getAddQuestionsBizId)
                    .collect(Collectors.toList());
            List<AddQuestions> addQuestionsList = iAddQuestionsService.queryList(addQuestionsBizIdList);
            if (!CollectionUtils.isEmpty(addQuestionsList)) {
                apiAddQuestionsDtoList = addQuestionsList.stream().map(d -> {
                    ApiAddQuestionsDto apiAddQuestionsDto = new ApiAddQuestionsDto();
                    BeanUtils.copyProperties(d,apiAddQuestionsDto);
                    return apiAddQuestionsDto;
                }).collect(Collectors.toList());
            }
        }
        return apiAddQuestionsDtoList;
    }

    /**
     * 选项附加问题列表
     * @param questionnaireBizId
     * @param questionBizId
     * @param optionBizId
     * @return
     */
    public List<ApiAddQuestionsDto> addQuestionsDtoList(String questionnaireBizId,
                                                        String questionBizId,
                                                        String optionBizId) {
        List<ApiAddQuestionsDto> apiAddQuestionsDtoList = new ArrayList<>();
        List<RelOptionsAddQuestions> relOptionsAddQuestionsList = iRelOptionsAddQuestionsService.queryList(questionnaireBizId,questionBizId,optionBizId);
        if (!CollectionUtils.isEmpty(relOptionsAddQuestionsList)) {
            List<String> addQuestionsBizIdList = relOptionsAddQuestionsList.stream()
                    .map(RelOptionsAddQuestions::getAddQuestionsBizId)
                    .collect(Collectors.toList());
            List<AddQuestions> addQuestionsList = iAddQuestionsService.queryList(addQuestionsBizIdList);
            if (!CollectionUtils.isEmpty(addQuestionsList)) {
                apiAddQuestionsDtoList = addQuestionsList.stream().map(d -> {
                    ApiAddQuestionsDto apiAddQuestionsDto = new ApiAddQuestionsDto();
                    BeanUtils.copyProperties(d,apiAddQuestionsDto);
                    return apiAddQuestionsDto;
                }).collect(Collectors.toList());
            }
        }
        return apiAddQuestionsDtoList;
    }

    /**
     * 查询当前问题回答的内容对象
     * @param objectBizId 答题的对象业务ID
     * @param questionnaireBizId 答题选择的问卷唯一业务ID
     * @param questionBizId 答题选择的问题唯一业务ID
     * @return
     */
    public ApiAnswerSessionsDto answerSessionsDto(String objectBizId,
                                                  String questionnaireBizId,
                                                  String questionBizId){
        ApiAnswerSessionsDto apiAnswerSessionsDto = null;
        //获取登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //查询回答列表
        List<AnswerSessions> answerSessionsList = iAnswerSessionsService.queryList(AnswerSessionsDto
                .builder()
                .questionnaireBizId(questionnaireBizId)
                .objectBizId(objectBizId)
                .questionBizId(questionBizId)
                .creatorId(authUserDto.getUserBizId())
                .build());
        if (!CollectionUtils.isEmpty(answerSessionsList)) {
            //获取回答对象
            AnswerSessions answerSessions = answerSessionsList.get(0);
            apiAnswerSessionsDto = new ApiAnswerSessionsDto();
            BeanUtils.copyProperties(answerSessions,apiAnswerSessionsDto);
            if (StringUtils.isNotBlank(apiAnswerSessionsDto.getOptionJson())) {
                List<ApiOptionJsonDto> optionJsonDtoList = JSON.parseArray(apiAnswerSessionsDto.getOptionJson(),ApiOptionJsonDto.class);
                apiAnswerSessionsDto.setOptionJsonDtoList(optionJsonDtoList);
            }
            if (StringUtils.isNotBlank(apiAnswerSessionsDto.getQuestionTextJson())) {
                List<ApiQuestionTextJsonDto> questionTextJsonDtoList = JSON.parseArray(apiAnswerSessionsDto.getQuestionTextJson(),ApiQuestionTextJsonDto.class);
                apiAnswerSessionsDto.setQuestionTextJsonDtoList(questionTextJsonDtoList);
            }
        }
        return apiAnswerSessionsDto;
    }

    /**
     * 问卷-答题提交接口
     * @return
     */
    @Override
    public Result<ApiAnswerSaveResponse> answerSave(ApiAnswerSaveRequest request) {
        if (CollectionUtils.isEmpty(request.getAnswerSessionsDtoList())) {
            //答题会话对象不能为空
            throw new BusinessException("答题会话对象集合不能为空！");
        }
        //校验问卷信息是否存在
        Result<Questionnaires> result = checkQuestionnairesIsExist(request.getQuestionnaireBizId());
        Questionnaires questionnaires = result.getData();
        //获取登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //先删后增加答题会话记录
        iAnswerSessionsService.del(AnswerSessionsDto
                .builder()
                .creatorId(authUserDto.getUserBizId())
                .objectBizId(request.getObjectBizId())
                .questionnaireBizId(request.getQuestionnaireBizId())
                .build());

        //增加
        if (!CollectionUtils.isEmpty(request.getAnswerSessionsDtoList())) {
            List<AnswerSessions> saveList = request.getAnswerSessionsDtoList().stream().map(dto -> {
                AnswerSessions answerSessions = new AnswerSessions();
                BeanUtils.copyProperties(dto,answerSessions);
                //答题选择的选项json
                answerSessions.setOptionJson(JSON.toJSONString(dto.getOptionJsonDtoList()));
                //问题的答题内容（问答题）json
                answerSessions.setQuestionTextJson(JSON.toJSONString(dto.getQuestionTextJsonDtoList()));
                //创建人ID
                answerSessions.setCreatorId(authUserDto.getUserBizId());
                answerSessions.setObjectBizId(request.getObjectBizId());
                answerSessions.setQuestionnaireBizId(request.getQuestionnaireBizId());
                //答题会话业务ID
                answerSessions.setSessionId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_ANSWER_SESSIONS.getCode()));
                //答题状态 1-已完成
                answerSessions.setStatus(1);
                answerSessions.setId(null);
                return answerSessions;
            }).collect(Collectors.toList());

            iAnswerSessionsService.saveOrUpdateBatch(saveList);
        }

        return Result.success();
    }

    /**
     * 问卷-业务对象绑定
     * @return
     */
    @Override
    public Result objectSave(ApiObjectSaveRequest request) {
        RelObjectQuestionnaire relObjectQuestionnaire = new RelObjectQuestionnaire();
        BeanUtils.copyProperties(request,relObjectQuestionnaire);
        iRelObjectQuestionnaireService.saveOrUpdate(relObjectQuestionnaire);
        return Result.success();
    }

    /**
     * 校验问卷信息是否存在
     * @param questionnaireBizId
     * @return
     */
    public Result<Questionnaires> checkQuestionnairesIsExist(String questionnaireBizId) {
        Questionnaires questionnaires = iQuestionnairesService.queryOne(questionnaireBizId);
        if (Objects.isNull(questionnaires)) {
            //当前数据不存在
            throw new BusinessException("问卷信息不存在！");
        }
        return Result.success(questionnaires);
    }
}
