package com.yd.user.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.user.api.service.ApiRelProjectRoleService;
import com.yd.user.api.service.ApiSysRoleService;
import com.yd.user.api.service.ApiRelTenantRoleService;
import com.yd.user.feign.request.sysrole.ApiSysRoleAddRequest;
import com.yd.user.feign.request.sysrole.ApiSysRoleEditRequest;
import com.yd.user.feign.request.sysrole.ApiSysRolePageRequest;
import com.yd.user.feign.response.sysrole.ApiSysRoleDetailResponse;
import com.yd.user.feign.response.sysrole.ApiSysRolePageResponse;
import com.yd.user.service.dto.RelTenantProjectDto;
import com.yd.user.service.dto.SysRoleDto;
import com.yd.user.service.model.RelTenantProject;
import com.yd.user.service.model.SysProject;
import com.yd.user.service.model.SysRole;
import com.yd.user.service.model.SysTenant;
import com.yd.user.service.service.IRelTenantProjectService;
import com.yd.user.service.service.ISysProjectService;
import com.yd.user.service.service.ISysRoleService;
import com.yd.user.service.service.ISysTenantService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
public class ApiSysRoleServiceImpl implements ApiSysRoleService {

    @Autowired
    private ISysRoleService iSysRoleService;

    @Autowired
    private IRelTenantProjectService iRelTenantProjectService;

    @Autowired
    private ISysProjectService iSysProjectService;

    @Autowired
    private ISysTenantService iSysTenantService;

    @Autowired
    private ApiRelTenantRoleService apiRelTenantRoleService;

    @Autowired
    private ApiRelProjectRoleService apiRelProjectRoleService;

    /**
     * 角色分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiSysRolePageResponse>> page(ApiSysRolePageRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;
        //当前所属的租户id（当前页面右上角显示的租户） 超级管理员不传值，非超级管理员根据传值来
        if (!isAdmin && StringUtils.isBlank(request.getLoginTenantBizId())) {
            //非超级管理员并且LoginTenantBizId为空，提示必须传值LoginTenantBizId
            throw new BusinessException("非超级管理员，当前登录用户所属的租户id（当前页面右上角显示的租户）必须传值");
        }
        String loginTenantBizId = isAdmin ? "" : request.getLoginTenantBizId();

        Page<ApiSysRolePageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        List<String> projectBizIdList = new ArrayList<>();

        if (!isAdmin) {
            //查询当前租户下绑定的项目列表id
            List<RelTenantProject> relTenantProjectList = iRelTenantProjectService.queryList(RelTenantProjectDto.builder().tenantBizId(loginTenantBizId).build());
            if (CollectionUtils.isEmpty(relTenantProjectList)) {
                projectBizIdList = relTenantProjectList.stream().map(RelTenantProject::getProjectBizId).collect(Collectors.toList());
            }
        }

        IPage<ApiSysRolePageResponse> iPage = iSysRoleService.page(page, request,isAdmin,projectBizIdList,loginTenantBizId);
        return Result.success(iPage);
    }

    /**
     * 获取角色详情
     * @param roleBizId 角色唯一标识(业务ID)
     * @return
     */
    @Override
    public Result<ApiSysRoleDetailResponse> detail(String roleBizId) {
        Result<SysRole> result = checkSysRoleIsExist(roleBizId);
        SysRole sysRole = result.getData();

        ApiSysRoleDetailResponse response = new ApiSysRoleDetailResponse();
        BeanUtils.copyProperties(sysRole,response);

        if (CommonEnum.SCOPE_SYS.getCode().equals(response.getScope().toString())) {
            //系统级
            response.setTenantBizId("");
            response.setTenantName("");
            response.setProjectBizId("");
            response.setProjectName("");
        }else if (CommonEnum.SCOPE_TENANT.getCode().equals(response.getScope().toString())) {
            //租户级
            SysTenant sysTenant = iSysTenantService.queryOne(response.getTenantBizId());
            if (!Objects.isNull(sysTenant)) {
                response.setTenantName(sysTenant.getTenantName());
            }
            response.setProjectBizId("");
            response.setProjectName("");
        }else if (CommonEnum.SCOPE_PROJECT.getCode().equals(response.getScope().toString())) {
            //项目级
            SysProject sysProject = iSysProjectService.queryOne(response.getProjectBizId());
            if (!Objects.isNull(sysProject)) {
                response.setProjectName(sysProject.getProjectName());
            }
            response.setTenantBizId("");
            response.setTenantName("");
        }
        return Result.success(response);
    }

    /**
     * 添加角色
     * @param request
     * @return
     */
    @Override
    public Result add(ApiSysRoleAddRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;

        //添加和编辑校验入参
        addAndEditReqCheck(isAdmin,
                request.getScope(),
                request.getTenantBizId(),
                request.getProjectBizId());

        //校验角色名称的唯一性
        List<SysRole> list = iSysRoleService.queryList(SysRoleDto.builder().roleName(request.getRoleName()).build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.ROLE_NAME_EXISTS.getCode(),ResultCode.ROLE_NAME_EXISTS.getMessage());
        }
        //新增角色
        SysRole sysRole = new SysRole();
        BeanUtils.copyProperties(request,sysRole);
        sysRole.setRoleBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_ROLE.getCode()));
        sysRole = addAndEditSaveSetCommon(sysRole);
        iSysRoleService.saveOrUpdate(sysRole);

        //新增或者编辑-更新租户校色关系、更新项目角色关系公共方法
        addAndEditByRelCommon(request.getTenantBizId(),
                request.getProjectBizId(),
                sysRole.getRoleBizId(),
                request.getScope(),
                CommonEnum.OPR_SOURCE_ADD.getCode());

        return Result.success();
    }

    /**
     * 编辑角色
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiSysRoleEditRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;

        //添加和编辑校验入参
        addAndEditReqCheck(isAdmin,
                request.getScope(),
                request.getTenantBizId(),
                request.getProjectBizId());

        //校验角色是否存在
        Result<SysRole> result = checkSysRoleIsExist(request.getRoleBizId());
        SysRole sysRole = result.getData();

        //校验角色名称的唯一性
        List<SysRole> list = iSysRoleService.queryList(SysRoleDto.builder()
                .roleName(request.getRoleName())
                .roleBizId(request.getRoleBizId())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.ROLE_NAME_EXISTS.getCode(),ResultCode.ROLE_NAME_EXISTS.getMessage());
        }

        //TODO 换绑逻辑（作用域换绑：提示校验需要先把什么解绑再来换绑）

        //编辑角色
        BeanUtils.copyProperties(request,sysRole);
        sysRole = addAndEditSaveSetCommon(sysRole);
        iSysRoleService.saveOrUpdate(sysRole);

        //新增或者编辑-更新租户校色关系、更新项目角色关系公共方法 TODO
//        addAndEditByRelCommon(request.getTenantBizId(),
//                request.getProjectBizId(),
//                sysRole.getRoleBizId(),
//                request.getScope(),
//                CommonEnum.OPR_SOURCE_ADD.getCode());

        return Result.success();
    }

    /**
     * 编辑角色状态
     * @param roleBizId 角色唯一标识(业务ID)
     * @param status 状态(0:禁用 1:启用)
     * @return
     */
    @Override
    public Result editStatus(String roleBizId, Integer status) {
        //校验角色是否存在
        Result<SysRole> result = checkSysRoleIsExist(roleBizId);
        SysRole sysRole = result.getData();

        sysRole.setStatus(status);
        iSysRoleService.saveOrUpdate(sysRole);
        return Result.success();
    }

    /**
     * 添加和编辑保存设置参数公共方法
     * @param sysRole
     * @return
     */
    public SysRole addAndEditSaveSetCommon(SysRole sysRole) {
        if (CommonEnum.SCOPE_SYS.getCode().equals(sysRole.getScope().toString())) {
            //系统级
            sysRole.setTenantBizId("");
            sysRole.setProjectBizId("");
        }else if (CommonEnum.SCOPE_TENANT.getCode().equals(sysRole.getScope().toString())) {
            //租户级
            sysRole.setProjectBizId("");
        }else if (CommonEnum.SCOPE_PROJECT.getCode().equals(sysRole.getScope().toString())) {
            //项目级
            sysRole.setTenantBizId("");
        }
        return sysRole;
    }

    /**
     * 新增或者编辑-更新租户角色关系、更新项目角色关系公共方法
     * @param tenantBizId 租户id
     * @param projectBizId 项目id
     * @param roleBizId 角色id
     * @param scope 作用域
     * @param oprSource 操作来源 1-添加 2-编辑
     */
    public void addAndEditByRelCommon(String tenantBizId,
                                      String projectBizId,
                                      String roleBizId,
                                      Integer scope,
                                      String oprSource) {
        if (CommonEnum.OPR_SOURCE_ADD.getCode().equals(oprSource)) {
            //添加来源
            if (CommonEnum.SCOPE_TENANT.getCode().equals(scope.toString())) {
                //作用域租户级-新增租户角色关系
                apiRelTenantRoleService.addSingleRel(tenantBizId,roleBizId);
            }
            if (CommonEnum.SCOPE_PROJECT.getCode().equals(scope.toString())) {
                //作用域项目级-新增项目角色关系
                apiRelProjectRoleService.addSingleRel(projectBizId,roleBizId);
            }
        }else if (CommonEnum.OPR_SOURCE_EDIT.getCode().equals(oprSource)) {
            //编辑来源 TODO 编辑
        }
    }

    /**
     * 添加和编辑校验入参
     * @param isAdmin
     * @param scope
     * @param tenantBizId
     * @param projectBizId
     */
    public void addAndEditReqCheck(Boolean isAdmin,
                                   Integer scope,
                                   String tenantBizId,
                                   String projectBizId) {
        if (!isAdmin && CommonEnum.SCOPE_SYS.getCode().equals(scope.toString())) {
            //非超级管理员不能选择系统级作用域
            throw new BusinessException("非超级管理员不能选择系统级作用域！");
        }
        //校验作用域scope字段，如果为作用域租户级所属租户id必须传值
        if(CommonEnum.SCOPE_TENANT.getCode().equals(scope.toString())
                && StringUtils.isBlank(tenantBizId)) {
            throw new BusinessException("租户级作用域，所属租户id不能为空");
        }
        //校验作用域scope字段，如果为作用域项目级所属项目id必须传值
        if(CommonEnum.SCOPE_PROJECT.getCode().equals(scope.toString())
                && StringUtils.isBlank(projectBizId)) {
            throw new BusinessException("项目级作用域，所属项目id不能为空");
        }
    }

    /**
     * 校验角色是否存在
     * @param roleBizId
     * @return
     */
    public Result<SysRole> checkSysRoleIsExist(String roleBizId) {
        SysRole sysRole = iSysRoleService.queryOne(roleBizId);
        if (Objects.isNull(sysRole)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(sysRole);
    }
}
