package com.yd.user.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.user.api.dto.ApiSysProjectAddAndEditReqCheckDto;
import com.yd.user.api.service.ApiRelTenantProjectService;
import com.yd.user.api.service.ApiSysProjectService;
import com.yd.user.feign.request.sysproject.ApiSysProjectAddRequest;
import com.yd.user.feign.request.sysproject.ApiSysProjectEditRequest;
import com.yd.user.feign.request.sysproject.ApiSysProjectPageRequest;
import com.yd.user.feign.response.sysproject.ApiSysProjectDetailResponse;
import com.yd.user.feign.response.sysproject.ApiSysProjectPageResponse;
import com.yd.user.service.dto.RelTenantProjectDto;
import com.yd.user.service.dto.SysProjectDto;
import com.yd.user.service.model.RelTenantProject;
import com.yd.user.service.model.SysProject;
import com.yd.user.service.model.SysTenant;
import com.yd.user.service.service.IRelTenantProjectService;
import com.yd.user.service.service.ISysProjectService;
import com.yd.user.service.service.ISysTenantService;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;

@Service
public class ApiSysProjectServiceImpl implements ApiSysProjectService {

    @Autowired
    private ISysProjectService iSysProjectService;

    @Autowired
    private ISysTenantService iSysTenantService;

    @Autowired
    private IRelTenantProjectService iRelTenantProjectService;

    @Autowired
    private ApiRelTenantProjectService apiRelTenantProjectService;

    /**
     * 项目分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiSysProjectPageResponse>> page(ApiSysProjectPageRequest request) {
        Page<ApiSysProjectPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;
        //当前所属的租户id（当前页面右上角显示的租户） 超级管理员不传值，非超级管理员根据传值来
        if (!isAdmin && StringUtils.isBlank(request.getLoginTenantBizId())) {
            //非超级管理员并且LoginTenantBizId为空，提示必须传值LoginTenantBizId
            throw new BusinessException("非超级管理员，当前登录用户所属的租户id（当前页面右上角显示的租户）必须传值");
        }
        String loginTenantBizId = isAdmin ? "" : request.getLoginTenantBizId();

        IPage<ApiSysProjectPageResponse> iPage = iSysProjectService.page(page, request,isAdmin,loginTenantBizId);
        return Result.success(iPage);
    }

    /**
     * 获取项目详情
     * @param projectBizId 项目唯一标识(业务ID)
     * @return
     */
    @Override
    public Result<ApiSysProjectDetailResponse> detail(String projectBizId) {
        SysProject sysProject = iSysProjectService.queryOne(projectBizId);
        if (Objects.isNull(sysProject)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        ApiSysProjectDetailResponse response = new ApiSysProjectDetailResponse();
        BeanUtils.copyProperties(sysProject,response);
        if (StringUtils.isNotBlank(sysProject.getTenantBizId())) {
            //租户id不为空，查名称
            SysTenant sysTenant = iSysTenantService.queryOne(sysProject.getTenantBizId());
            if (!Objects.isNull(sysTenant)) {
                response.setTenantName(sysTenant.getTenantName());
            }
        }
        return Result.success(response);
    }

    /**
     * 添加项目
     * @param request
     * @return
     */
    @Override
    public Result add(ApiSysProjectAddRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;

        ApiSysProjectAddAndEditReqCheckDto checkDto = new ApiSysProjectAddAndEditReqCheckDto();
        BeanUtils.copyProperties(request,checkDto);
        checkDto.setIsAdmin(isAdmin);
        //添加和编辑校验入参
        addAndEditReqCheck(checkDto);

        //校验项目名称的唯一性
        List<SysProject> list = iSysProjectService.queryList(SysProjectDto.builder()
                .projectName(request.getProjectName())
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            //项目名称已存在
            throw new BusinessException(ResultCode.PROJECT_NAME_EXISTS.getCode(),ResultCode.PROJECT_NAME_EXISTS.getMessage());
        }

        //新增项目基础信息
        SysProject sysProject = new SysProject();
        BeanUtils.copyProperties(request,sysProject);
        sysProject.setProjectBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_PROJECT.getCode()));
        //添加和编辑保存设置参数公共方法
        sysProject = addAndEditSaveSetCommon(sysProject);
        iSysProjectService.saveOrUpdate(sysProject);

        if(CommonEnum.SCOPE_TENANT.getCode().equals(request.getScope().toString())){
            //租户级作用域，新增租户和项目之间的关系，系统级作用域不用新增（只有在租户管理配置项目导入的时候新增租户项目关系）
            apiRelTenantProjectService.addSingleRel(request.getTenantBizId(),sysProject.getProjectBizId());
        }

        return Result.success();
    }

    /**
     * 编辑项目
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiSysProjectEditRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;

        ApiSysProjectAddAndEditReqCheckDto checkDto = new ApiSysProjectAddAndEditReqCheckDto();
        BeanUtils.copyProperties(request,checkDto);
        checkDto.setIsAdmin(isAdmin);
        //添加和编辑校验入参
        addAndEditReqCheck(checkDto);

        //校验项目是否存在·
        Result<SysProject> result = checkSysProjectIsExist(request.getProjectBizId());
        SysProject sysProject = result.getData();

        //校验项目名称的唯一性
        List<SysProject> list = iSysProjectService.queryList(SysProjectDto.builder()
                .projectName(request.getProjectName())
                .projectBizId(request.getProjectBizId())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            //项目名称已存在
            throw new BusinessException(ResultCode.PROJECT_NAME_EXISTS.getCode(),ResultCode.PROJECT_NAME_EXISTS.getMessage());
        }

        //编辑-作用域字段和所属租户改变校验  TODO 编辑作用域和所属租户的校验待定
//        changeCheck(sysProject.getScope(),request.getScope(),request.getProjectBizId());

        BeanUtils.copyProperties(request,sysProject);
        //添加和编辑保存设置参数公共方法
        sysProject = addAndEditSaveSetCommon(sysProject);
        iSysProjectService.saveOrUpdate(sysProject);

        return Result.success();
    }

    /**
     * 添加和编辑保存设置参数公共方法
     * @param sysProject
     * @return
     */
    public SysProject addAndEditSaveSetCommon(SysProject sysProject) {
        if (CommonEnum.SCOPE_SYS.getCode().equals(sysProject.getScope().toString())) {
            sysProject.setTenantBizId("");
        }
        return sysProject;
    }

    /**
     * 添加和编辑校验入参
     * @param checkDto
     */
    public void addAndEditReqCheck(ApiSysProjectAddAndEditReqCheckDto checkDto) {
        if (!checkDto.getIsAdmin() && CommonEnum.SCOPE_SYS.getCode().equals(checkDto.getScope().toString())) {
            //非超级管理员不能选择系统级作用域
            throw new BusinessException("非超级管理员不能选择系统级作用域！");
        }
        //校验作用域scope字段，如果为作用域租户级所属租户id必须传值
        if(CommonEnum.SCOPE_TENANT.getCode().equals(checkDto.getScope().toString())
                && StringUtils.isBlank(checkDto.getTenantBizId())) {
            throw new BusinessException("租户级作用域，所属租户id不能为空");
        }

        //校验内置项目字段，如果不是内置项目，项目访问地址必须传值
        if (checkDto.getIsIn() == 0 && StringUtils.isBlank(checkDto.getProjectUrl())) {
            throw new BusinessException("不是内置项目，项目访问地址不能为空");
        }
    }

    /**
     * 编辑项目状态
     * @param projectBizId 项目唯一标识(业务ID)
     * @param status 状态(0:禁用 1:启用)
     * @return
     */
    @Override
    public Result editStatus(String projectBizId, Integer status) {
        //校验项目是否存在
        Result<SysProject> result = checkSysProjectIsExist(projectBizId);
        SysProject sysProject = result.getData();

        sysProject.setStatus(status);
        iSysProjectService.saveOrUpdate(sysProject);

        return Result.success();
    }


    /**
     * 编辑-作用域和所属租户改变校验
     * @param oldScope 旧值（表里数据）
     * @param newScope 新值（请求入参值）
     * @param projectBizId 项目id
     */
    public void changeCheck(Integer oldScope,Integer newScope,String projectBizId){
        //TODO 发生改变暂时提示不让编辑这个作用域和所属租户
        if (!oldScope.equals(newScope)) {
            //发生改变，查询当前项目有没有绑定的租户关系，有则提示不能编辑作用域字段
            List<RelTenantProject> list = iRelTenantProjectService.queryList(RelTenantProjectDto.builder().projectBizId(projectBizId).build());
            if (!CollectionUtils.isEmpty(list)) {
                throw new BusinessException("当前项目下有绑定的租户关系，不能编辑作用域选项！");
            }
        }
    }

    /**
     * 校验项目是否存在
     * @param projectBizId
     * @return
     */
    public Result<SysProject> checkSysProjectIsExist(String projectBizId) {
        SysProject sysProject = iSysProjectService.queryOne(projectBizId);
        if (Objects.isNull(sysProject)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(sysProject);
    }

}
