package com.yd.user.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ProjectEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.feign.client.agentreferralrelation.ApiAgentReferralRelationFeignClient;
import com.yd.csf.feign.request.agentreferralrelation.ApiAgentReferralRelationSaveRequest;
import com.yd.user.api.service.ApiClientUserService;
import com.yd.user.feign.request.clientuser.ApiClientUserPageRequest;
import com.yd.user.feign.response.clientuser.ApiClientUserDetailResponse;
import com.yd.user.feign.response.clientuser.ApiClientUserPageResponse;
import com.yd.user.feign.response.sysuser.ApiSysUserPageResponse;
import com.yd.user.service.dto.ClientUserDto;
import com.yd.user.service.dto.SysUserDto;
import com.yd.user.service.model.ClientUser;
import com.yd.user.service.model.SysUser;
import com.yd.user.service.service.IClientUserService;
import com.yd.user.service.service.ISysUserService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

@Service
public class ApiClientUserServiceImpl implements ApiClientUserService {

    @Autowired
    private ISysUserService iSysUserService;

    @Autowired
    private IClientUserService iClientUserService;

    @Autowired
    private ApiAgentReferralRelationFeignClient apiAgentReferralRelationFeignClient;

    /**
     * 分页查询-客户端用户信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiClientUserPageResponse>> page(ApiClientUserPageRequest request) {
        Page<ApiClientUserPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiClientUserPageResponse> iPage = iClientUserService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 详情-客户端用户信息
     * @param clientUserBizId
     * @return
     */
    @Override
    public Result<ApiClientUserDetailResponse> detail(String clientUserBizId) {
        Result<ClientUser> result = checkClientUserIsExist(clientUserBizId);
        ClientUser clientUser = result.getData();
        ApiClientUserDetailResponse response = new ApiClientUserDetailResponse();
        BeanUtils.copyProperties(clientUser,response);
        return Result.success(response);
    }

    /**
     * 同步sys_user系统用户，批量更新客户端用户表数据
     * @param userBizIdList
     * @param tenantBizId
     * @param projectBizId
     * @return
     */
    @Override
    public Result saveClientUserList(List<String> userBizIdList,
                                     String tenantBizId,
                                     String projectBizId) {
        List<SysUser> sysUserList = iSysUserService.queryList(SysUserDto.builder().userBizIdList(userBizIdList).build());
        if (!CollectionUtils.isEmpty(sysUserList)) {
            //根据用户唯一标识(sys_user系统用户表)列表查询客户端用户列表，来做更新。
            List<ClientUser> clientUserList = iClientUserService.queryList(ClientUserDto.builder()
                    .tenantBizId(tenantBizId)
                    .projectBizId(projectBizId)
                    .userBizIdList(userBizIdList)
                    .build());
            List<ClientUser> saveList = new ArrayList<>();
            if (CollectionUtils.isEmpty(clientUserList)) {
                //全部新增同步sys_user数据
                saveList = buildClientUserList(sysUserList,tenantBizId,projectBizId);
            }else {
                //部分新增部分更新
                //过滤出不在clientUserList中的sysUser
                Set<String> existUserBizIds = clientUserList.stream()
                        .map(ClientUser::getUserBizId)
                        .collect(Collectors.toSet());
                List<SysUser> newSysUsers = sysUserList.stream()
                        .filter(sysUser -> !existUserBizIds.contains(sysUser.getUserBizId()))
                        .collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(newSysUsers)) {
                    //新增同步sys_user数据
                    saveList = buildClientUserList(sysUserList,tenantBizId,projectBizId);
                }
            }

            if (!CollectionUtils.isEmpty(saveList)) {
                //批量保存客户端用户数据
                iClientUserService.saveOrUpdateBatch(saveList);

                //批量保存推荐关系
                saveAgentReferralRelation(tenantBizId,projectBizId,saveList);
            }
        }
        return Result.success();
    }

    /**
     * 根据List<SysUser>构造List<ClientUser>
     * @param sysUserList
     * @return
     */
    public List<ClientUser> buildClientUserList(List<SysUser> sysUserList,
                                                String tenantBizId,
                                                String projectBizId) {
        List<ClientUser> clientUserList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(sysUserList)) {
            clientUserList = sysUserList.stream().map(dto -> {
                ClientUser clientUser = new ClientUser();
                //待审核
                clientUser.setApprovalStatus(CommonEnum.APPROVAL_STATUS_DSH.getCode());
                clientUser.setClientUserBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_CLIENT_USER.getCode()));
                clientUser.setEmail(dto.getEmail());
                //邀请码 TODO
//                clientUser.setInvitationCode();
                clientUser.setName(dto.getRealName());
                clientUser.setPhone(dto.getMobile());
                clientUser.setTenantBizId(tenantBizId);
                clientUser.setProjectBizId(projectBizId);
                //用户唯一标识(sys_user系统用户表)
                clientUser.setUserBizId(dto.getUserBizId());
                //注册来源 - PC
                clientUser.setRegisterSource(CommonEnum.SOURCE_TYPE_PC.getCode());
                return clientUser;
            }).collect(Collectors.toList());
        }
        return clientUserList;
    }

    /**
     * 保存推荐关系（目前只有CSF-客户端项目有这个添加，后期有其他项目再加）
     * @return
     */
    public Result saveAgentReferralRelation(String tenantBizId,
                                            String projectBizId,
                                            List<ClientUser> clientUserList) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        if (ProjectEnum.CSF_MINI_PROGRAM.getProjectBizId().equals(projectBizId)
                && !CollectionUtils.isEmpty(clientUserList)) {
            //CSF小程序项目 - 添加推荐关系
            ApiAgentReferralRelationSaveRequest request = new ApiAgentReferralRelationSaveRequest();
            List<String> agentIdList = clientUserList.stream().map(ClientUser::getClientUserBizId).collect(Collectors.toList());
            request.setAgentIdList(agentIdList);
            request.setProjectBizId(projectBizId);
            //推荐人ID - 当前登录人用户ID
            request.setReferrerId(authUserDto.getUserBizId());
            request.setSource(CommonEnum.SOURCE_TYPE_PC.getCode());
            request.setTenantBizId(tenantBizId);
            request.setReferrerAccount(authUserDto.getUsername());
            apiAgentReferralRelationFeignClient.save(request);
        }
        return Result.success();
    }

    /**
     * 根据条件查询客户端用户业务ID列表
     * @param tenantBizId
     * @param tenantBizId
     * @return
     */
    @Override
    public Result<List<String>> clientUserBizIdList(String tenantBizId, String projectBizId) {
        List<String> clientUserBizIdList = new ArrayList<>();
        List<ClientUser> clientUserList = iClientUserService.queryList(ClientUserDto.builder()
                .tenantBizId(tenantBizId)
                .projectBizId(projectBizId)
                .build());
        if (!CollectionUtils.isEmpty(clientUserList)) {
            clientUserBizIdList = clientUserList.stream().map(ClientUser::getClientUserBizId).collect(Collectors.toList());
        }
        return Result.success(clientUserBizIdList);
    }

    /**
     * 校验客户端用户是否存在
     * @param clientUserBizId
     * @return
     */
    public Result<ClientUser> checkClientUserIsExist(String clientUserBizId) {
        ClientUser clientUser = iClientUserService.queryOne(clientUserBizId);
        if (Objects.isNull(clientUser)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(clientUser);
    }
}
