package com.yd.user.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.TreeUtils;
import com.yd.user.api.service.ApiRelTenantMenuService;
import com.yd.user.feign.request.reltenantmenu.ApiRelTenantMenuListAddRequest;
import com.yd.user.feign.request.reltenantmenu.ApiRelTenantMenuListPageRequest;
import com.yd.user.feign.request.reltenantmenu.ApiRelTenantMenuImportQueryMenuTreeRequest;
import com.yd.user.feign.request.reltenantmenu.ApiRelTenantMenuTreeListRequest;
import com.yd.user.feign.response.reltenantmenu.ApiRelTenantMenuListPageResponse;
import com.yd.user.feign.response.reltenantmenu.ApiRelTenantMenuImportQueryMenuTreeResponse;
import com.yd.user.feign.response.reltenantmenu.ApiRelTenantMenuTreeListResponse;
import com.yd.user.feign.response.sysmenu.ApiSysMenuTreeListResponse;
import com.yd.user.service.dto.RelTenantMenuDto;
import com.yd.user.service.model.RelTenantMenu;
import com.yd.user.service.model.SysMenu;
import com.yd.user.service.service.IRelTenantMenuService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
public class ApiRelTenantMenuServiceImpl implements ApiRelTenantMenuService {

    @Autowired
    private IRelTenantMenuService iRelTenantMenuService;

    /**
     * 租户和菜单关系分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiRelTenantMenuListPageResponse>> page(ApiRelTenantMenuListPageRequest request) {
        Page<ApiRelTenantMenuListPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiRelTenantMenuListPageResponse> iPage = iRelTenantMenuService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 租户和菜单关系树形列表查询
     * @param request
     * @return
     */
    @Override
    public Result<List<ApiRelTenantMenuTreeListResponse>> tree(ApiRelTenantMenuTreeListRequest request) {
        List<ApiRelTenantMenuTreeListResponse> treeMenu = new ArrayList<>();
        //租户和菜单关系列表
        List<ApiRelTenantMenuTreeListResponse> queryList = iRelTenantMenuService.treeList(request);

        if (!CollectionUtils.isEmpty(queryList)) {
            //构建菜单树形结构
            treeMenu = TreeUtils.buildTree(
                    queryList,
                    ApiRelTenantMenuTreeListResponse::getMenuBizId,         // 获取节点ID
                    ApiRelTenantMenuTreeListResponse::getParentBizId,   // 获取父节点ID
                    (parent, children) -> parent.setChildren(children), // 设置子节点
                    "0"                   // 根节点的父ID值
            );
        }
        return Result.success(treeMenu);
    }

    /**
     * 添加租户和菜单列表关系
     * @param request
     * @return
     */
    @Override
    public Result addMenuList(ApiRelTenantMenuListAddRequest request) {
        List<RelTenantMenu> list = new ArrayList<>();
        //先删后增加
        iRelTenantMenuService.removeByTenantBizId(request.getTenantBizId());

        if (!CollectionUtils.isEmpty(request.getMenuBizIdList())) {
            //菜单唯一标识(业务ID) 列表（选中列表）不为空
            list = request.getMenuBizIdList().stream().map(bizId -> {
                RelTenantMenu relTenantMenu = new RelTenantMenu();
                relTenantMenu.setTenantBizId(request.getTenantBizId());
                relTenantMenu.setMenuBizId(bizId);
                return relTenantMenu;
            }).collect(Collectors.toList());

            iRelTenantMenuService.saveOrUpdateBatch(list);
        }

        return Result.success();
    }

    /**
     * 删除租户和菜单关系
     * @param id 租户和菜单关系表主键id
     * @return
     */
    @Override
    public Result del(Long id) {
        RelTenantMenu relTenantMenu = iRelTenantMenuService.queryOne(id);
        if (Objects.isNull(relTenantMenu)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        iRelTenantMenuService.removeById(relTenantMenu.getId());
        return Result.success();
    }

    /**
     * 导入-树形菜单查询
     * @param request
     * @return
     */
    @Override
    public Result<List<ApiRelTenantMenuImportQueryMenuTreeResponse>> importQueryMenuTree(ApiRelTenantMenuImportQueryMenuTreeRequest request) {
        //TODO 条件搜索菜单树后续完善
        List<ApiRelTenantMenuImportQueryMenuTreeResponse> treeMenu = new ArrayList<>();
        //查询租户和菜单关系数据
        List<ApiRelTenantMenuImportQueryMenuTreeResponse> queryList = iRelTenantMenuService.importQueryMenuTreeList(request);

        if (!CollectionUtils.isEmpty(queryList)) {
            //构建菜单树形结构
            treeMenu = TreeUtils.buildTree(
                    queryList,
                    ApiRelTenantMenuImportQueryMenuTreeResponse::getMenuBizId,         // 获取节点ID
                    ApiRelTenantMenuImportQueryMenuTreeResponse::getParentBizId,   // 获取父节点ID
                    (parent, children) -> parent.setChildren(children), // 设置子节点
                    "0"                   // 根节点的父ID值
            );
        }

        return Result.success(treeMenu);
    }

    /**
     * 导入-树形菜单-查询选中的菜单id集合
     * @param tenantBizId
     * @return
     */
    @Override
    public Result<List<String>> selectedList(String tenantBizId) {
        List<String> menuBizIdList = new ArrayList<>();
        List<RelTenantMenu> relTenantMenuList = iRelTenantMenuService.queryList(RelTenantMenuDto.builder()
                .tenantBizId(tenantBizId)
                .build());
        if (!CollectionUtils.isEmpty(relTenantMenuList)) {
            menuBizIdList = relTenantMenuList.stream().map(RelTenantMenu::getMenuBizId).collect(Collectors.toList());
        }
        return Result.success(menuBizIdList);
    }

    /**
     * 添加单个租户和菜单关系
     * @param tenantBizId
     * @param menuBizId
     * @return
     */
    @Override
    public Result addSingleRel(String tenantBizId, String menuBizId) {
        //新增关系
        RelTenantMenu relTenantMenu = new RelTenantMenu();
        relTenantMenu.setTenantBizId(tenantBizId);
        relTenantMenu.setMenuBizId(menuBizId);
        iRelTenantMenuService.saveOrUpdate(relTenantMenu);
        return Result.success();
    }
}
